#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "main.h"

struct mem_region * memory_region[MAX_MEM_REGIONS];

unsigned int stack_start = 0x80000000;

/* declare machine state and registers */
unsigned int R[32]; /* CPU registers */
unsigned int CpCond[4], CCR[4][32], CPR[4][32];
float FGR[16];      /* FPU registers */
double *FPR = (void *) FGR;
int *FWR = (void *) FGR;

// Other global variables
unsigned int entry_point = 0;

// Print register values
void print_registers() {
  int i;
  for (i=0; i<32; i++) {
    printf("R[%02d]=0x%08x%s", i, R[i], (i+1)%4 ? " " : "\n");
  }
  printf("\n");
}

// Print the top N elements of the stack
void print_stack() {

  int i;

  printf("Stack pointer is: 0x%08x\n", R[REG_SP]);

  for (i=(opt_printstack-1); i>=0; i--) {
    if ( R[REG_SP]+(i*4) < stack_start+STACK_SIZE) {
      printf("stack[0x%08x]=0x%08x\n", R[REG_SP]+(i*4), _mem_read_word(R[REG_SP]+(i*4)));
    }
  }
  
}

// Convert simulated MIPS address to a pointer on the host machine
void* sim_to_real_addr(unsigned int addr) {

  int i;

  for (i=0; i<MAX_MEM_REGIONS; i++) {
    if (!memory_region[i]) break;
    if ((addr >= memory_region[i]->addr) &&
        (addr < memory_region[i]->addr + memory_region[i]->size)) {
      return (void*)(memory_region[i]->ptr + addr - memory_region[i]->addr);
    }
  }

  fprintf(stderr, "Error: Address 0x%08x is out of range!\n", addr);
  exit(1);
  return NULL;

}

// Print memory regions for debugging
void debug_print_regions() {
  int i;
  for (i=0; i<MAX_MEM_REGIONS; i++) {
    if (!memory_region[i]) break;
    printf("region=%d  addr=0x%08x  size=0x%08x\n",
      i, memory_region[i]->addr, memory_region[i]->size);
  }
}

// Read a word from MIPS address space
unsigned int _mem_read_word(unsigned int addr) {
  unsigned int* value = (unsigned int*)sim_to_real_addr(addr);
  return *value;
}

// Read a half word from MIPS address space
unsigned short _mem_read_half(unsigned int addr) {
  unsigned short* value = (unsigned short*)sim_to_real_addr(addr);
  return *value;
}

// Read a byte from MIPS address space
unsigned char _mem_read_byte(unsigned int addr) {
  unsigned char* value = (unsigned char*)sim_to_real_addr(addr);
  return *value;
}

// Write a word to MIPS address space
void _mem_write_word(unsigned int addr, unsigned int value) {
  unsigned int* val = (unsigned int*)sim_to_real_addr(addr);
  *val = value;
}

// Write a half word to MIPS address space
void _mem_write_half(unsigned int addr, unsigned short value) {
  unsigned short* val = (unsigned short*)sim_to_real_addr(addr);
  *val = value;
}

// Write a byte to MIPS address space
void _mem_write_byte(unsigned int addr, unsigned char value) {
  unsigned char* val = (unsigned char*)sim_to_real_addr(addr);
  *val = value;
}

// Initialize memory
void initialize_memory() {

  int i;

  for (i=0; i<32; i++) {
    R[i] = 0;
  }

  for (i=0; i<MAX_MEM_REGIONS; i++) {
    memory_region[i] = NULL;
  }

}

// Allocate an initialize the MIPS stack
void create_stack() {

  int i;
  char * netid;
  char netid_hash[4];
  void * netid_hash_ptr;
  unsigned int stack_offset;

  // Determine the stack location as a function of the student's NetID
  netid = getenv("NETID");
  if (!netid) {
    netid = getenv("USER");
    if (!netid) {
      fprintf(stderr, "Your environment does not contain a $NETID or $USER variable\n");
      exit(1);
    }
  }
  printf("NetID: %s\n", netid);
  memset(netid_hash, 0, 4);
  for (i=0; i<4; i++) {
    if (netid[i]=='\0') break;
    netid_hash[i] = (char)tolower(netid[i]);
  }
  netid_hash_ptr = netid_hash;
  stack_offset = (unsigned int)(*(unsigned int *)netid_hash_ptr);
  stack_offset >>= 2;
  stack_offset &= 0xfffffff8; // ensure proper alignment
  //printf("Stack offset: %d\n", stack_offset);
  stack_start = 0x40000000 + stack_offset;
  //printf("Starting stack at: 0x%08x\n", stack_start);

  for (i=0;i<MAX_MEM_REGIONS;i++) {
    if (memory_region[i]==NULL) break;
  }

  memory_region[i] = malloc(sizeof(struct mem_region));
  if (!memory_region[i]) {
    perror("cannot allocate memory for stack");
    exit(1);
  }

  memory_region[i]->ptr = malloc(STACK_SIZE);
  if (!memory_region[i]->ptr) {
    perror("cannot allocate memory for stack");
    exit(1);
  }

  memory_region[i]->addr = stack_start;
  memory_region[i]->size = STACK_SIZE;

  R[REG_SP] = memory_region[i]->addr + memory_region[i]->size - 4;

}
