#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"

int opt_disassemble=0, opt_printstack=0, opt_dumpreg=0;
unsigned int opt_cache_associativity=2, opt_cache_size=8, opt_block_size=4;

void syntax_error() {
  printf("Usage: simulate [-r] [-d] [-s <N>] <filename>\n");
  printf("  -r      Dump registers at each instruction\n");
  printf("  -d      Disassemble each instruction\n");
  printf("  -s <N>  Show the top <N> integers on the stack at each instruction\n");
  printf("  -S <N>  Use an <N>-way set associative cache\n");
  printf("  -N <N>  Use an <N>-block cache\n");
  printf("  -bs <N> Use <N> words per block\n");
  printf("  -pc     Print cache statistics when complete\n");
  printf("  -dc     Dump cache contents when complete\n");
  exit(1);
}

void print_cache() {
  unsigned int hits = cache_hits();
  unsigned int misses = cache_misses();
  unsigned int reads = cache_reads();
  unsigned int writes = cache_writes();
  float rate = cache_read_hit_rate();

  printf("Cache Statistics:\n");
  printf("--------------------\n");
  printf("Reads   : %10i\n",reads);
  printf("Writes  : %10i\n",writes);
  printf("Hits    : %10i\n",hits);
  printf("Misses  : %10i\n",misses);
  printf("Hit Rate: %10.3f\n",rate);
  
}

void dump_cache() {
  unsigned int i;
  unsigned int j;
  int tag;

  printf("\n");
  for (i = 0; i < opt_cache_size / opt_cache_associativity; ++i)
  {
    for (j = 0; j < opt_cache_associativity; ++j)
    {
      tag = probe_cache(i, j);
      if (tag == -1)
        printf("________ ");
      else
        printf("%08x ", tag);
    }
    printf("\n");
  }
}

int main(int argc, char **argv) {
  char* filename = NULL;
  int i;
  int p_cache = 0;
  int d_cache = 0;

  for (i=1; i<argc; i++) {
    if (!strcmp("-d", argv[i])) {
      opt_disassemble = 1;
    } else if (!strcmp("-r", argv[i])) {
      opt_dumpreg = 1;
    } else if (!strcmp("-s", argv[i])) {
      i++;
      if (i<argc) {
        if ( sscanf(argv[i], "%i", &opt_printstack) != 1 ) {
          syntax_error();
        }
      }
    } else if (!strcmp("-S", argv[i])) {
      i++;
      if (i<argc) {
        if ( sscanf(argv[i], "%u", &opt_cache_associativity) != 1 ) {
          syntax_error();
        }
      }
    } else if (!strcmp("-bs", argv[i])) {
      i++;
      if (i<argc) {
        if ( sscanf(argv[i], "%u", &opt_block_size) != 1 ) {
          syntax_error();
        }
      }
    } 
    else if (!strcmp("-N", argv[i])) {
      i++;
      if (i<argc) {
        if ( sscanf(argv[i], "%u", &opt_cache_size) != 1 ) {
          syntax_error();
        }
      }
    } else if(!strcmp("-pc", argv[i])) {
      p_cache = 1;
    } else if(!strcmp("-dc", argv[i])) {
      d_cache = 1;
    } else if (i == (argc-1)) {
      filename = argv[i];
    } else {
      syntax_error();
    }
  }

  if (!filename) syntax_error();

  if (opt_cache_associativity == 0)
    opt_cache_associativity = opt_cache_size;
  else if (opt_cache_associativity > opt_cache_size)
  {
    fprintf(stderr, "Set associativity must be less than cache size.\n");
    syntax_error();
  }
  if (opt_cache_size % opt_cache_associativity != 0)
  {
    fprintf(stderr, "Cache size must evenly divide set associativity.\n");
    syntax_error();
  }

  i = 1;
  do
  {
    if (opt_cache_size / opt_cache_associativity == i)
      break;
    i <<= 1;
  } while (i != 0);
  if (i == 0)
  {
    fprintf(stderr, "Cache size over associativity must be a power of two.\n");
    syntax_error();
  }

  initialize_memory();

  initialize_cache(opt_cache_associativity, opt_cache_size, opt_block_size);

  readelf(filename);

  create_stack();

  run();

  printf("\n");
  if(p_cache)
    print_cache();
  if(d_cache)
    dump_cache();
  free_cache();

  return 0;

}
