/* Copyright (c) 2006, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.gui.menu;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import com.cburch.logisim.circuit.Circuit;
import com.cburch.logisim.file.LoadedLibrary;
import com.cburch.logisim.file.Loader;
import com.cburch.logisim.file.LogisimFile;
import com.cburch.logisim.proj.Project;
import com.cburch.logisim.tools.AddTool;
import com.cburch.logisim.tools.Library;
import com.cburch.logisim.tools.Tool;

public class Popups {
    private static class ProjectPopup extends JPopupMenu
            implements ActionListener {
        Project proj;
        JMenuItem add = new JMenuItem(Strings.get("projectAddCircuitItem"));
        JMenu load = new JMenu(Strings.get("projectLoadLibraryItem"));
        JMenuItem loadBuiltin = new JMenuItem(Strings.get("projectLoadBuiltinItem"));
        JMenuItem loadLogisim = new JMenuItem(Strings.get("projectLoadLogisimItem"));
        JMenuItem loadJar = new JMenuItem(Strings.get("projectLoadJarItem"));

        ProjectPopup(Project proj) {
            super(Strings.get("projMenu"));
            this.proj = proj;

            load.add(loadBuiltin); loadBuiltin.addActionListener(this);
            load.add(loadLogisim); loadLogisim.addActionListener(this);
            load.add(loadJar); loadJar.addActionListener(this);

            add(add); add.addActionListener(this);
            add(load);
        }

        public void actionPerformed(ActionEvent e) {
            Object src = e.getSource();
            if(src == add) {
                ProjectCircuitActions.doAddCircuit(proj);
            } else if(src == loadBuiltin) {
                ProjectLibraryActions.doLoadBuiltinLibrary(proj);
            } else if(src == loadLogisim) {
                ProjectLibraryActions.doLoadLogisimLibrary(proj);
            } else if(src == loadJar) {
                ProjectLibraryActions.doLoadJarLibrary(proj);
            }
        }
    }

    private static class LibraryPopup extends JPopupMenu
            implements ActionListener {
        Project proj;
        Library lib;
        JMenuItem unload = new JMenuItem(Strings.get("projectUnloadLibraryItem"));
        JMenuItem reload = new JMenuItem(Strings.get("projectReloadLibraryItem"));

        LibraryPopup(Project proj, Library lib, boolean is_top) {
            super(Strings.get("libMenu"));
            this.proj = proj;
            this.lib = lib;

            add(unload); unload.addActionListener(this);
            add(reload); reload.addActionListener(this);
            unload.setEnabled(is_top);
            reload.setEnabled(is_top && lib instanceof LoadedLibrary);
        }

        public void actionPerformed(ActionEvent e) {
            Object src = e.getSource();
            if(src == unload) {
                ProjectLibraryActions.doUnloadLibrary(proj, lib);
            } else if(src == reload) {
                Loader loader = proj.getLogisimFile().getLoader();
                loader.reload((LoadedLibrary) lib);
            }
        }
    }

    private static class CircuitPopup extends JPopupMenu
            implements ActionListener {
        Project proj;
        Tool tool;
        Circuit circuit;
        JMenuItem view = new JMenuItem(Strings.get("projectViewCircuitItem"));
        JMenuItem analyze = new JMenuItem(Strings.get("projectAnalyzeCircuitItem"));
        JMenuItem rename = new JMenuItem(Strings.get("projectRenameCircuitItem"));
        JMenuItem main = new JMenuItem(Strings.get("projectSetAsMainItem"));
        JMenuItem remove = new JMenuItem(Strings.get("projectRemoveCircuitItem"));

        CircuitPopup(Project proj, Tool tool, Circuit circuit) {
            super(Strings.get("circuitMenu"));
            this.proj = proj;
            this.tool = tool;
            this.circuit = circuit;

            add(view); view.addActionListener(this);
            add(analyze); analyze.addActionListener(this);
            addSeparator();
            add(rename); rename.addActionListener(this);
            add(main); main.addActionListener(this);
            add(remove); remove.addActionListener(this);
            
            boolean canChange = proj.getLogisimFile().contains(circuit);
            LogisimFile file = proj.getLogisimFile();
            view.setEnabled(proj.getCurrentCircuit() != circuit);
            rename.setEnabled(canChange);
            main.setEnabled(canChange && file.getMainCircuit() != circuit);
            remove.setEnabled(canChange && file.getCircuitCount() > 1
                    && proj.getDependencies().canRemove(circuit));
        }

        public void actionPerformed(ActionEvent e) {
            Object source = e.getSource();
            if(source == view) {
                proj.setCurrentCircuit(circuit);
            } else if(source == analyze) {
                ProjectCircuitActions.doAnalyze(proj, circuit);
            } else if(source == rename) {
                ProjectCircuitActions.doRenameCircuit(proj, circuit);
            } else if(source == main) {
                ProjectCircuitActions.doSetAsMainCircuit(proj, circuit);
            } else if(source == remove) {
                ProjectCircuitActions.doRemoveCircuit(proj, circuit);
            }
        }
    }
    
    public static JPopupMenu forCircuit(Project proj, AddTool tool, Circuit circ) {
        return new CircuitPopup(proj, tool, circ);
    }
    
    public static JPopupMenu forTool(Project proj, Tool tool) {
        return null;
    }
    
    public static JPopupMenu forProject(Project proj) {
        return new ProjectPopup(proj);
    }
    
    public static JPopupMenu forLibrary(Project proj, Library lib, boolean isTop) {
        return new LibraryPopup(proj, lib, isTop);
    }

}
