/* Copyright (c) 2006, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.circuit;

import java.util.ArrayList;
import java.util.Iterator;

public class Simulator {
    private class PropagationManager extends Thread {
        private Propagator propagator = null;
        private int ticksRequested = 0;
        private boolean resetRequested = false;
        private boolean propagateRequested = false;
        private boolean complete = false;
        
        public Propagator getPropagator() {
            return propagator;
        }
        
        public void setPropagator(Propagator value) {
            propagator = value;
        }
        
        public synchronized void requestPropagate() {
            if(!propagateRequested) {
                propagateRequested = true;
                notifyAll();
            }
        }
        
        public synchronized void requestReset() {
            if(!resetRequested) {
                resetRequested = true;
                notifyAll();
            }
        }
        
        public synchronized void requestTick()  {
            ticksRequested++;
            notifyAll();
        }
        
        public synchronized void shutDown() {
            complete = true;
            notifyAll();
        }
        
        public void run() {
            while(!complete) {
                synchronized(this) {
                    while(!complete && !propagateRequested
                            && !resetRequested && ticksRequested == 0) {
                        try {
                            wait();
                        } catch(InterruptedException e) { }
                    }
                }

                if(resetRequested) {
                    resetRequested = false;
                    if(propagator != null) propagator.reset();
                    firePropagationCompleted();
                }
                
                if(propagateRequested || ticksRequested > 0) {
                    boolean ticked = false;
                    propagateRequested = false;
                    if(isRunning && propagator != null) {
                        ticked = ticksRequested > 0;
                        if(ticked) {
                            synchronized(this) {
                                ticksRequested--;
                            }
                            propagator.tick();
                        }
                        do {
                            propagateRequested = false;
                            try {
                                exceptionEncountered = false;
                                propagator.propagate();
                            } catch(Throwable thr) {
                                thr.printStackTrace();
                                exceptionEncountered = true;
                                setIsRunning(false);
                            }
                        } while(propagateRequested);
                        if(isOscillating()) setIsRunning(false);
                    } else if(isRunning) { // propagator == null
                        ticksRequested = 0;
                    }
                    if(ticked) fireTickCompleted();
                    firePropagationCompleted();
                }
            }
        }
    }
    
    private class Ticker extends Thread {
        private boolean shouldTick = false;
        private int ticksPending = 0;
        private boolean complete = false;
        
        public synchronized void shutDown() {
            complete = true;
            notifyAll();
        }

        public void run() {
            long lastTick = System.currentTimeMillis();
            while(true) {
                boolean curShouldTick = shouldTick;
                int curTickFrequency = tickFrequency;
                try {
                    synchronized(this) {
                        curShouldTick = shouldTick;
                        curTickFrequency = tickFrequency;
                        while(!curShouldTick && ticksPending == 0
                                && !complete) {
                            wait();
                            curShouldTick = shouldTick;
                            curTickFrequency = tickFrequency;
                        }
                    }
                } catch(InterruptedException e) { }
                
                if(complete) break;

                long now = System.currentTimeMillis();
                if(ticksPending > 0 || (curShouldTick
                        && now >= lastTick + curTickFrequency)) {
                    lastTick = now;
                    manager.requestTick();
                    synchronized(this) {
                        if(ticksPending > 0) ticksPending--;
                    }
                    // we fire tickCompleted in this thread so that other
                    // objects (in particular the repaint process) can slow
                    // the thread down.
                }

                try {
                    long nextTick = lastTick + curTickFrequency;
                    int wait = (int) (nextTick - System.currentTimeMillis());
                    if(wait < 20) wait = 20;
                    if(wait > 100) wait = 100;
                    Thread.sleep(wait);
                } catch(InterruptedException e) { }
            }
        }

        synchronized void awake() {
            shouldTick = isRunning && isTicking && tickFrequency > 0;
            if(shouldTick) notifyAll();
        }
    }

    private boolean isRunning = true;
    private boolean isTicking = false;
    private boolean exceptionEncountered = false;
    private int tickFrequency = 250;

    private PropagationManager manager = new PropagationManager();
    private Ticker ticker = new Ticker();
    private ArrayList listeners = new ArrayList();
    private Thread propagationThread = null;
    private Object propagateLock = new Object();

    public Simulator() {
        try {
            manager.setPriority(manager.getPriority() - 1);
            ticker.setPriority(ticker.getPriority() - 1);
        } catch(SecurityException e) {
        } catch(IllegalArgumentException e) { }
        manager.start();
        ticker.start();
    }
    
    public void shutDown() {
        ticker.shutDown();
        manager.shutDown();
    }

    public void setCircuitState(CircuitState state) {
        manager.setPropagator(state.getPropagator());
        ticker.awake();
    }
    
    public CircuitState getCircuitState() {
        Propagator prop = manager.getPropagator();
        return prop == null ? null : prop.getRootState();
    }
    
    public void requestReset() {
        manager.requestReset();
    }
    
    public void tick() {
        synchronized(ticker) {
            ticker.ticksPending++;
            ticker.notifyAll();
        }
    }
    
    public boolean isExceptionEncountered() {
        return exceptionEncountered;
    }

    public boolean isRunning() {
        return isRunning;
    }

    public void setIsRunning(boolean value) {
        if(isRunning != value) {
            isRunning = value;
            ticker.awake();
            fireSimulatorStateChanged();
        }
    }

    public boolean isTicking() {
        return isTicking;
    }

    public void setIsTicking(boolean value) {
        if(isTicking != value) {
            isTicking = value;
            ticker.awake();
            fireSimulatorStateChanged();
        }
    }

    public int getTickFrequency() {
        return tickFrequency;
    }

    public void setTickFrequency(int millis) {
        if(tickFrequency != millis) {
            tickFrequency = millis;
            ticker.awake();
            fireSimulatorStateChanged();
        }
    }

    public void requestPropagate() {
        manager.requestPropagate();
    }

    public boolean isOscillating() {
        Propagator prop = manager.getPropagator();
        return prop != null && prop.isOscillating();
    }

    public void addSimulatorListener(SimulatorListener l) { listeners.add(l); }
    public void removeSimulatorListener(SimulatorListener l) { listeners.remove(l); }
    void firePropagationCompleted() {
        SimulatorEvent e = new SimulatorEvent(this);
        ArrayList listeners = new ArrayList(this.listeners);
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            SimulatorListener l = (SimulatorListener) it.next();
            l.propagationCompleted(e);
        }
    }
    void fireTickCompleted() {
        SimulatorEvent e = new SimulatorEvent(this);
        ArrayList listeners = new ArrayList(this.listeners);
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            ((SimulatorListener) it.next()).tickCompleted(e);
        }
    }
    void fireSimulatorStateChanged() {
        SimulatorEvent e = new SimulatorEvent(this);
        ArrayList listeners = new ArrayList(this.listeners);
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            ((SimulatorListener) it.next()).simulatorStateChanged(e);
        }
    }
}
