/*
 * Door.java
 *
 * This class is a ObstacleSprite referencing a rectangular obstacle such as a
 * crate or the "win door". All it does is override the constructor. We do this
 * for organizational purposes. Otherwise we have to put a lot of initialization
 * code in the scene, and that just makes the scene too long and unreadable.
 *
 * Based on the original PhysicsDemo Lab by Don Holden, 2007
 *
 * Author:  Walker M. White
 * Version: 2/8/2025
 */
package edu.cornell.cis3152.physics.rocket;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.utils.JsonValue;
import edu.cornell.gdiac.assets.ParserUtils;
import edu.cornell.gdiac.physics2.BoxObstacle;
import edu.cornell.gdiac.physics2.ObstacleSprite;

/**
 * A rectangular obstacle for the rocket demo.
 *
 * This class can represent a crate or a "win door". An ObstacleSprite is a
 * sprite (specifically a textured mesh) that is connected to a obstacle. It
 * is designed to be the same size as the physics object, and it tracks the
 * physics object, matching its positionand angle at all times.
 *
 * The reason we use a textured mesh instead of a image is because it allows
 * us more control over the size and shape of the image. We will talk about
 * how to use these later in class. For now, just notice how we create meshes.
 */
public class Box extends ObstacleSprite {

    /**
     * Creates a box with the given physics units and settings
     *
     * The physics units are used to size the mesh relative to the physics
     * body. All other attributes are defined by the JSON file
     *
     * @param units     The physics units
     * @param settings  The door physics constants
     */
    public Box(float x, float y, float units, JsonValue settings) {
        super();

        float s = settings.getFloat( "size" );
        float size = s*units;

        obstacle = new BoxObstacle(x, y, s, s);

        obstacle.setDensity( settings.getFloat( "density", 0 ) );
        obstacle.setFriction( settings.getFloat( "friction", 0 ) );
        obstacle.setRestitution( settings.getFloat( "restitution", 0 ) );
        obstacle.setPhysicsUnits( units );
        obstacle.setUserData( this );

        debug = ParserUtils.parseColor( settings.get("debug"),  Color.WHITE);

        // Create a rectangular mesh the same size as the door, adjusted by
        // the physics units. For all meshes attached to a physics body, we
        // want (0,0) to be in the center of the mesh. So the method call below
        // is (x,y,w,h) where x, y is the bottom left.
        mesh.set(-size/2.0f,-size/2.0f,size,size);
    }

}
