/*
 * Scene.java
 *
 * A scene corresponds to a player mode. We talked about a player mode in the
 * the first few weeks of class. A player mode is often like a self-contained
 * game in its own right. Therefore, it should have its own root (sub)controller.
 * Since the game will often switch back and forth between player scenes, we
 * would like the modes to have a uniform interface.  That way they can all be
 * handled the same by the true root controller, GDXRoot.
 *
 * @author: Walker M. White
 * @version: 2/2/2025
 */
package edu.cornell.cis3152.optimization;

import edu.cornell.gdiac.graphics.SpriteBatch;

/**
 * Interface for the root class of a scene (player mode)
 *
 * Each scene is a game to itself and so it needs an update and a draw cycle.
 * The class constructor allocates all necessary resources, but the class must
 * have an explicit dispose() to release resources (which do not depend on
 * garbage collection)
 */
public interface Scene {

    /**
     * Reads user input, calculates physics, and updates the models.
     *
     * This method is HALF of the basic game loop. Every graphics frame calls
     * the method update() and the method draw(). The method update() contains
     * all of the calculations for updating the world, such as checking for
     * collisions, gathering input, and playing audio. It should not contain
     * any calls for drawing to the screen.
     *
     * @param delta Number of seconds since last animation frame
     */
    public void update(float delta);

    /**
     * Draws the scene on to the provided sprite batch
     *
     * There should be no code in this method that alters the game state. All
     * assignments should be to local variables or cache fields only.
     *
     * @param batch The sprite batch
     */
    public void draw(SpriteBatch batch);

    /**
     * Dispose of all (non-static) resources allocated to this mode.
     */
    public void dispose();

    /**
     * Resizes the window for this player mode to the given dimensions.
     *
     * This method is not guaranteed to be called when the player mode starts.
     * If the window size is important to the player mode, then these values
     * should be passed to the constructor at start.
     *
     * @param width The width of the game window
     * @param height The height of the game window
     */
    public void resize(int width, int height);

    /**
     * Returns true if this scene has completed and wishes to exit
     *
     * @return true if this scene has completed and wishes to exit
     */
    public boolean exitScene();
}
