"""
This is a replacement for the flash tile viewer.

While it can be run on most copies of Python, it does require 
for you to install Pillow, the Python imaging module.

Author: Walker M. White
Data: February 3, 2022
"""
import tkinter as tk
from PIL import Image, ImageTk
import argparse


# This is a coded layout of the tile viewer
# Each two digits is a reference to a tile number
# A void is represented by 99
LAYOUT = """
00990102020399129999999999999999999999999999
99129999999999282903990107039999013115999999
99249912999901404199129936991299993739991315
99060246039999992499060399011999999999993039
99249936999901071799369912993699999912992499
99369999999999249999990118039999991343992815
99999901059904179912999999999999993739993739
12999999369936990117999999131599999999999999
16039999999999999999999913452799999999132903
99999999131599999913159937384231141599373999
99999999372003990108420599999937113999999999
99129999993699999936992499999999369999999999
13439999999999999999991631159999999999131599
25442903991315999912999925100399999901232799
25323999133439990109159937399999999999373999
37419999373999999937399999999999129999999999
99249999999999999999999999999913221599999999
01460399131415991314159999129937383999131599
99369999252627993733279913210399999999373515
99999999373839999937399937399999999999993739
"""

def parse_args():
    """
    Returns the command line arguments sent by the user
    """
    parser = argparse.ArgumentParser(description='View a tile source file.')
    parser.add_argument('file', type=str, help='the tile source file')
    parser.add_argument('-r', '--rows', type=int, default=4,
                        help='the number of rows in the tile source file')
    parser.add_argument('-c', '--cols', type=int, default=12,
                        help='the number of columns in the tile source file')
    parser.add_argument('-x', '--width', type=int, default=30,
                        help='the display width of each tile')
    parser.add_argument('-y', '--height', type=int, default=30,
                        help='the display height of each tile')
    args = parser.parse_args()
    return args


def decode(layout):
    """
    Returns a numerical encoding of the map layout
    
    The parameter layout is a string encoding of the map layout. It
    is a multiline string where each (nonempty) line is a row in the
    map. For each row, each pair of characters is a number representing
    a tile. The pair '99' represents a void.
    
    Parameter layout: The map layout
    Precondition: layout is a string
    """
    lines = layout.split('\n')
    result = []
    for code in lines:
        if code.strip() != '':
            row = []
            pref = ''
            for x in code:
                if pref == '':
                    pref = x
                else:
                    value = int(pref+x)
                    if value == 99:
                        value = None
                    row.append(value)
                    pref = ''
            result.append(row)
    return result


def tilize(image,rows,cols,width,height):
    """
    Returns a list of tiles from the given image.
    
    The image is cut up into pieces according to the number of rows and 
    columns.  Each tile is resized to have the given width and height.
   
    Parameter image: The image file to tilize
    Precondition: image is a string referencing an image file
   
    Parameter rows: The number of rows in image
    Precondition: rows is an int > 0 
    
    Parameter cols: The number of columns in image
    Precondition: cols is an int > 0 
    
    Parameter width: The width of each tile
    Precondition: width is an int > 0 
    
    Parameter height: The height of each tile
    Precondition: height is an int > 0
    """
    im = Image.open(image)
    (ow,oh) = im.size
    
    # Tile sizes
    tw = ow/cols
    th = oh/rows

    # Scaling factor
    sw = (width*cols)/ow
    sh = (height*rows)/oh
    
    result = []
    for y in range(rows):
        for x in range(cols):
            piece = im.crop((x*tw, y*th, x*tw+tw, y*th+th))
            piece = piece.resize((width,height), Image.ANTIALIAS)
            tile = ImageTk.PhotoImage(piece)
            result.append(tile)
    
    return result


def initialize(title):
    """
    Returns a root-level TK window
    
    Parameter title: The window title
    Precondition: title is a string
    """
    root = tk.Tk()
    root.title(title)
    return root


def populate(root,map,tiles):
    """
    Creates a tiled canvas in the given root
    
    The size of the canvas is determined by the tile size. It will be large
    enough to hold the layout defined by the map.
    
    Parameter root: The TK root window
    Precondition: root is a TK window
    
    Parameter map: The map encoding
    Precondition: map is a 2d array of ints 0..99
    
    Parameter tiles: The tile images
    Precondition: tiles is a list of PhotoImage objects
    """
    if len(tiles) == 0 or tiles[0] is None or len(map) == 0:
        return
    
    w = tiles[0].width()
    h = tiles[0].height()
    rows = len(map)
    cols = len(map[0])
    
    canvas = tk.Canvas(root, bg="white", height=(rows+2)*h, width=(cols+2)*w)
    canvas.pack()
    
    for y in range(len(map)):
        row = map[y]
        for x in range(len(row)):
            if not row[x] is None:
                image = tiles[row[x]]
                canvas.create_image(x*w+w, y*h+h, image=image, anchor="nw")


def main():
    """
    Runs the TK application
    
    This function will continue to run until the window is closed.
    """
    args = parse_args()
    root  = initialize("Tile Viewer")
    tiles = tilize(args.file,args.rows,args.cols,args.width,args.height)
    map = decode(LAYOUT)
    populate(root,map,tiles)
    root.mainloop()


if __name__ == '__main__':
    main()