/*
 * GameMode.java
 *
 * This is the primary class file for running the game. You should study this
 * file for ideas on how to structure your own root class. This class follows a
 * model-view-controller pattern fairly strictly.
 *
 * Based on original Parallax Sample by Don Holden, 2007
 *
 * @author: Walker M. White
 * @version, 2/15/2025
 */
package edu.cornell.cis3152.parallax;

import com.badlogic.gdx.math.Affine2;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.utils.JsonValue;
import edu.cornell.gdiac.assets.AssetDirectory;
import edu.cornell.gdiac.graphics.SpriteBatch;
import edu.cornell.gdiac.util.*;

import com.badlogic.gdx.*;
import com.badlogic.gdx.assets.*;
import com.badlogic.gdx.graphics.*;

/**
 * The primary controller class for the game.
 *
 * While GDXRoot is the root class, it delegates all of the work to the scene
 * classes. This is the scene class for running the game. In initializes all
 * of the other classes in the game and hooks them together. It also provides
 * the basic game loop (update-draw).
 */
public class GameScene implements Screen {
    /** The window width in pixels */
    protected float width;
    /** The window height in pixels */
    protected float height;

	// GRAPHICS
	// Asset loading is handled statically, so these are static variables
	/** The absolute background layer */
	private Texture sunsetTexture;
	/** The mountains in front of the sunset */
	private Texture mountainTexture;
	/** The road underneath our wheels */
	private Texture roadTexture;
	/** The grass layer */
	private Texture grassTexture;
	/** The car wheels */
	private Texture wheelTexture;
	/** The car body */
	private Texture chassisTexture;

	/** The data constants */
	private JsonValue constants;

	/** Reference to the underlying asset directory */
	private AssetDirectory directory;
    protected OrthographicCamera camera;
    /** Reference to the sprite batch */
    private SpriteBatch batch;
    /** The drawing transform */
    private final Affine2 transform = new Affine2();

	/** Reads input from keyboard or game pad (CONTROLLER CLASS) */
	private InputController inputController;

	/** Whether or not this player mode is still active */
	private boolean active;
	/** Listener that will update the player mode when we are done */
	private ScreenListener listener;

	/** Vehicle state */
    private float carPosition;
    private float wheelRotation;
    private float bumpSine;

	/**
	 * Creates a new game with the given drawing context.
	 *
	 * This constructor initializes the models and controllers for the game. The
	 * view has already been initialized by the root class.
	 */
	public GameScene(AssetDirectory directory, SpriteBatch batch) {
		this.batch = batch;
        grassTexture = directory.getEntry("grass",Texture.class);
        roadTexture  = directory.getEntry("road",Texture.class);
        sunsetTexture = directory.getEntry("sunset",Texture.class);
        mountainTexture = directory.getEntry("mountain",Texture.class);
        wheelTexture = directory.getEntry("wheels",Texture.class);
        chassisTexture = directory.getEntry("chassis",Texture.class);

        constants = directory.getEntry("constants",JsonValue.class);

		// Create the controllers.
		inputController = new InputController();
        resize( Gdx.graphics.getWidth(), Gdx.graphics.getHeight() );
	}

	/**
	 * Dispose of all (non-static) resources allocated to this mode.
	 */
	public void dispose() {
		inputController = null;
        directory = null;
		batch = null;
	}


	/**
	 * Update the game state.
	 *
	 * We prefer to separate update and draw from one another as separate
	 * methods, instead of using the single render() method that LibGDX does.
	 * We will talk about why we prefer this in lecture.
	 *
	 * @param delta Number of seconds since last animation frame
	 */
	private void update(float delta) {
		inputController.sync();

		// Allows the game to exit
		if (inputController.didExit()) {
			Gdx.app.exit();
		}

        // Change position of car
		carPosition += inputController.getSpeed();
		bumpSine += inputController.getSpeed() * constants.get("bumpiness").getFloat("frequency",0);

		// Change rotation in degrees
		float FULLCIRCLE = 360.0f;
		wheelRotation -= (inputController.getSpeed()*FULLCIRCLE) / (Math.PI*wheelTexture.getWidth());
		if (wheelRotation > FULLCIRCLE) {
			wheelRotation -= FULLCIRCLE;
		} else if (wheelRotation < 0.0f) {
			wheelRotation += FULLCIRCLE;
		}
	}

	/**
	 * Draws the parallax scene
	 *
	 * We prefer to separate update and draw from one another as separate
	 * methods, instead of using the single render() method that LibGDX does.
	 * We will talk about why we prefer this in lecture.
	 */
	private void draw(float delta) {
		batch.begin(camera);

        // Camera's X-position, 400 pixels (half the screen) to the left of the car.
		float parallax = -carPosition;

        // Draw the sunset
		drawWrapped(sunsetTexture,
                    constants.get("parallax").getFloat("sunset",0) * parallax,
                    constants.get("height").getInt("sunset",0));

        // Draw the mountains
		drawWrapped(mountainTexture,
                    constants.get("parallax").getFloat("mountains",0) * parallax,
                    constants.get("height").getInt("mountains",0));

        // Draw the background grass
		drawWrapped(grassTexture,
		            constants.get("parallax").getFloat("backgrass",0) * parallax,
                    constants.get("height").getInt("backgrass",0));

        // Draw the road
		drawWrapped(roadTexture,
		            constants.get("parallax").getFloat("road",0) * parallax,
                    constants.get("height").getInt("road",0));

        // Draw the wheels
        float ox = wheelTexture.getWidth()/2.0f;
        float oy = wheelTexture.getHeight()/2.0f;

        float offset;
        offset =  constants.get("parallax").getFloat("road",0) * constants.get("offset").getFloat("wheel1",0);
        SpriteBatch.computeTransform( transform, ox, oy, offset, constants.get("height").getInt("wheel",0), wheelRotation, 1.0f, 1.0f);
        batch.draw( wheelTexture, transform );

        offset =  constants.get("parallax").getFloat("road",0) * constants.get("offset").getFloat("wheel2",0);
        SpriteBatch.computeTransform( transform, ox, oy, offset, constants.get("height").getInt("wheel",0), wheelRotation, 1.0f, 1.0f);
		batch.draw(wheelTexture, transform);

        // Draw the car
        ox = chassisTexture.getWidth()/2.0f;
        oy = chassisTexture.getHeight()/2.0f;
        float bump = constants.get("bumpiness").getFloat("amplitude",0) * (float)Math.sin(bumpSine);
       	offset =  constants.get("parallax").getFloat("road",0) * constants.get("offset").getFloat("car",0);
        SpriteBatch.computeTransform( transform, ox, oy, offset, constants.get("height").getInt("car",0)+bump, 0.0f, 1.0f, 1.0f);
        batch.draw(chassisTexture, transform);

        drawWrapped(grassTexture,
                    constants.get("parallax").getFloat("foregrass",0) * parallax,
                    constants.get("height").getInt("foregrass",0));

		batch.end();
	}

    /**
     * Draws the seamless background image.
     *
     * The background image is drawn (with NO SCALING) at position x, y.
     * Width-wise, the image is seamlessly scrolled; when we reach the image
     * we draw a second copy.
     *
     * To work properly, the image should be wide enough to fill the screen.
     * However, it can be of any height.
     *
     * @param image Texture to draw as an overlay
     * @param x 	The x-coordinate of the bottom left corner
     * @param y 	The y-coordinate of the bottom left corner
     */
    public void drawWrapped(Texture image, float x, float y) {
        float wx = x;
        if (wx < 0) {
            float n = (float)Math.floor(-x/ width);
            wx += (1 + n) * width;
        } else if (wx > 0) {
            wx %= width;
        }

        // Have to draw the background twice for continuous scrolling.
        batch.draw(image, wx, y);
        batch.draw(image, wx-width, y);
    }

	/**
	 * Called when the Screen is resized.
	 *
	 * This can happen at any point during a non-paused state but will never
	 * happen before a call to show().
	 *
	 * @param width  The new width in pixels
	 * @param height The new height in pixels
	 */
	public void resize(int width, int height) {
        this.width  = width;
        this.height = height;
        if (camera == null) {
            camera = new OrthographicCamera();
        }
        camera.setToOrtho( false, width, height );
	}

	/**
	 * Called when the Screen should render itself.
	 *
	 * We defer to the other methods update() and draw().  However, it is VERY
	 * important that we only quit AFTER a draw.
	 *
	 * @param delta Number of seconds since last animation frame
	 */
	public void render(float delta) {
		if (active) {
			update(delta);
			draw(delta);
			if (inputController.didExit() && listener != null) {
				listener.exitScreen(this, 0);
			}
		}
	}

	/**
	 * Called when the Screen is paused.
	 *
	 * This is usually when it's not active or visible on screen. An Application
	 * is also paused before it is destroyed.
	 */
	public void pause() {
		// TODO Auto-generated method stub
	}

	/**
	 * Called when the Screen is resumed from a paused state.
	 *
	 * This is usually when it regains focus.
	 */
	public void resume() {
		// TODO Auto-generated method stub
	}

	/**
	 * Called when this screen becomes the current screen for a Game.
	 */
	public void show() {
		// Useless if called in outside animation loop
		active = true;
	}

	/**
	 * Called when this screen is no longer the current screen for a Game.
	 */
	public void hide() {
		// Useless if called in outside animation loop
		active = false;
	}

	/**
	 * Sets the ScreenListener for this mode
	 *
	 * The ScreenListener will respond to requests to quit.
	 */
	public void setScreenListener(ScreenListener listener) {
		this.listener = listener;
	}
}
