/*
 * Wall.java
 *
 * This class is a ObstacleSprite referencing either a wall. All it does is
 * override the constructor. We do this for organizational purposes. Otherwise
 * we have to put a lot of initialization code in the scene, and that just
 * makes the scene too long and unreadable.
 *
 * Based on the original PhysicsDemo Lab by Don Holden, 2007
 *
 * Author:  Walker M. White
 * Version: 3/1/2025
 */
 package edu.cornell.cis3152.cube;

import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.math.Affine2;
import com.badlogic.gdx.math.Rectangle;
import com.badlogic.gdx.physics.box2d.BodyDef;
import com.badlogic.gdx.utils.JsonValue;
import edu.cornell.gdiac.assets.ParserUtils;
import edu.cornell.gdiac.math.Poly2;
import edu.cornell.gdiac.math.PolyTriangulator;
import edu.cornell.gdiac.physics2.BoxObstacle;
import edu.cornell.gdiac.physics2.ObstacleSprite;
import edu.cornell.gdiac.physics2.PolygonObstacle;

/**
 * A class representing a tiled swall
 *
 * An ObstacleSprite is a sprite (specifically a textured mesh) that is
 * connected to a obstacle. It is designed to be the same size as the physics
 * object, and it tracks the physics object, matching its position and angle at
 * all times.
 *
 * This class demonstrates WHY we use meshes, even though we did not use them
 * in earlier labs. For a surface, we do not want to draw a simple rectangular
 * image. This time we want to tile a texture on a polygonal shape. Creating
 * such tiles is something the designers had to do in Lab 2. The nice thing
 * about ObstacleSprite, is that you can have a mesh with the exact same shape
 * as the physics body (adjusted for physics units), and then apply a texture
 * to that shape.
 */
public class Wall extends ObstacleSprite {

    /**
     * Creates a surface from the given set of points
     *
     * The points are in box2d space, not drawing space. They will be scaled
     * by the physics units when draw (which are determined by the settings).
     * The points define the outline of the shape. To work correctly, the points
     * must be specified in counterclockwise order, and the line segments may
     * not cross.
     *
     * @param points    The outline of the shape as x,y pairs
     * @param data      The physics constants for this rope bridge
     */
    public Wall(float[] points, JsonValue settings) {
        super();

        float sceneHeight = settings.get("scene_size").getFloat(1);
        float worldHeight = settings.get("world_size").getFloat(1);
        float units = sceneHeight/worldHeight;

        float tile = settings.getFloat( "tile" );

        // Construct a Poly2 object, breaking it into triangles
        Poly2 poly = new Poly2();
        PolyTriangulator triangulator = new PolyTriangulator();
        triangulator.set(points);
        triangulator.calculate();
        triangulator.getPolygon(poly);

        obstacle = new PolygonObstacle(points);
        obstacle.setBodyType( BodyDef.BodyType.StaticBody );
        obstacle.setDensity( settings.getFloat( "density", 0 ) );
        obstacle.setFriction( settings.getFloat( "friction", 0 ) );
        obstacle.setRestitution( settings.getFloat( "restitution", 0 ) );
        obstacle.setPhysicsUnits( units );
        debug = ParserUtils.parseColor( settings.get("debug"),  Color.WHITE);

        // Scale the polygon up and create a tiled mesh
        poly.scl( units );
        mesh.set(poly,tile,tile);
    }

}
