/*
 * InputController.java
 *
 * This class buffers in input from the devices and converts it into its
 * semantic meaning. If your game had an option that allows the player to
 * remap the control keys, you would store this information in this class.
 * That way, the main GameEngine does not have to keep track of the current
 * key mapping.
 *
 * Based on the original PhysicsDemo Lab by Don Holden, 2007
 *
 * Author:  Walker M. White
 * Version: 3/1/2025
 */
package edu.cornell.cis3152.cube;

import com.badlogic.gdx.*;

/**
 * Class for reading player input.
 *
 * This demo just detects keyboard input.
 */
public class InputController {
	/** The singleton instance of the input controller */
	private static InputController theController = null;

	/**
	 * Return the singleton instance of the input controller
	 *
	 * @return the singleton instance of the input controller
	 */
	public static InputController getInstance() {
		if (theController == null) {
			theController = new InputController();
		}
		return theController;
	}

	// Fields to manage buttons
	/** Whether the reset button was pressed. */
	private boolean leftPressed;
	/** Whether the right arrow was pressed. */
	private boolean rightPressed;
	/** Whether the exit button was pressed. */
	private boolean exitPressed;
	private boolean exitPrevious;

	/**
	 * Returns true if the left arrow was pressed.
	 *
	 * @return true if the left arrow was pressed.
	 */
	public boolean left() {
		return leftPressed;
	}

	/**
	 * Returns true if the right arrow was pressed.
	 *
	 * @return true if the right arrow was pressed.
	 */
	public boolean right() {
		return rightPressed;
	}

	/**
	 * Returns true if the exit button was pressed.
	 *
	 * @return true if the exit button was pressed.
	 */
	public boolean exit() {
		return exitPressed && !exitPrevious;
	}

	/**
	 * Creates a new input controller
	 *
	 * This version of the input controller only reads from a keyboard.
	 */
	public InputController() {
	}

	/**
	 * Syncs the keyboard to the current animation frame.
	 *
	 * Keyboard timing does not have to be aligned to animation frames. This method
	 * takes a snapshot at the beginning of the animation frame to record the
	 * current keyboard state.
	 */
	public void sync() {
		// Copy state from last animation frame
		// Helps us ignore buttons that are held down
		exitPrevious = exitPressed;
		leftPressed = Gdx.input.isKeyPressed(Input.Keys.LEFT);
		rightPressed = Gdx.input.isKeyPressed(Input.Keys.RIGHT);
		exitPressed  = Gdx.input.isKeyPressed(Input.Keys.ESCAPE);
	}
}
