/*
 * GDXRoot.java
 *
 * This is the primary class file for running the game.  It is the "static main"
 * of LibGDX. Typically an application is a subclass of Game that manages
 * multiple Screens, one of each scene.
 *
 * Based on the original PhysicsDemo Lab by Don Holden, 2007
 *
 * Author:  Walker M. White
 * Version: 3/1/2025
 */
package edu.cornell.cis3152.cube;

import com.badlogic.gdx.*;
import edu.cornell.gdiac.graphics.*;
import edu.cornell.gdiac.util.*;
import edu.cornell.gdiac.assets.*;

/**
 * Root class for a LibGDX.
 *
 * This class is technically not the ROOT CLASS. Each platform has another class
 * above this (e.g. PC games use DesktopLauncher) which serves as the true root.
 * However,  those classes are unique to each platform, while this class is the
 * same across all plaforms. Therefore, this class functions as the root class
 * for all intents and purposes, and you would draw it as a root class in an
 * architecture specification.
 */
public class GDXRoot extends Game implements ScreenListener {
	/** AssetManager to load game assets (textures, sounds, etc.) */
	AssetDirectory directory;
	/** The spritebatch to draw the screen (VIEW CLASS) */
	private SpriteBatch batch;
	/** Scene for the asset loading screen (CONTROLLER CLASS) */
	private LoadingScene loading;
	/** The scene for the cube demo (CONTROLLER CLASS) */
	private CubeScene cube;

	/**
	 * Creates a new game from the configuration settings.
	 *
	 * This method configures the asset manager, but does not load any assets
	 * or assign any screen.
	 */
	public GDXRoot() { }

	/**
	 * Called when the Application is first created.
	 *
	 * This is method immediately loads assets for the loading screen, and \
	 * prepares the asynchronous loader for all other assets.
	 */
	public void create() {
		batch  = new SpriteBatch();
		loading = new LoadingScene("jsons/assets.json" ,batch,1);

		// Initialize physics controller
		loading.setScreenListener(this);
		setScreen(loading);
	}

	/**
	 * Called when the Application is destroyed.
	 *
	 * This is preceded by a call to pause().
	 */
	public void dispose() {
		// Call dispose on our children
		setScreen(null);
		if (loading != null) {
			loading.dispose();
			loading = null;
		}
		if (cube != null) {
			cube.dispose();
			cube = null;
		}
		batch.dispose();
		batch = null;

		// Unload all of the resources
		if (directory != null) {
			directory.unloadAssets();
			directory.dispose();
			directory = null;
		}
		super.dispose();
	}

	/**
	 * Called when the Application is resized.
	 *
	 * This can happen at any point during a non-paused state but will never happen
	 * before a call to create().
	 *
	 * @param width  The new width in pixels
	 * @param height The new height in pixels
	 */
	public void resize(int width, int height) {
		if (loading != null) {
			loading.resize(width,height);
		}
		if (cube != null) {
			cube.resize( width, height );
		}
		super.resize(width,height);
	}

	/**
	 * The given screen has made a request to exit its player mode.
	 *
	 * The value exitCode can be used to implement menu options.
	 *
	 * @param screen   The screen requesting to exit
	 * @param exitCode The state of the screen upon exit
	 */
	public void exitScreen(Screen screen, int exitCode) {
		if (screen == loading) {
			cube = new CubeScene(loading.getAssets(), batch);
			//boxes.setScreenListener(this);
			setScreen(cube);
			loading.dispose();
			loading = null;
		} else {
			// We quit the main application
			Gdx.app.exit();
		}
	}

}
