/*
 * Run.java
 *
 * This class represents choosing a tree and moving towards it. It is the second
 * part of the "pee sequence". It is adapted from the GDX AI example written by
 * davebaol.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
 package edu.cornell.cis3152.behavior.tasks;

import com.badlogic.gdx.ai.GdxAI;
import com.badlogic.gdx.ai.btree.Task;
import com.badlogic.gdx.ai.btree.LeafTask;
import com.badlogic.gdx.math.Vector2;
import edu.cornell.cis3152.behavior.Dog;

/**
 * A task to target a tree and move towards it.
 *
 * This class is very similar to {@link PlayTask} except that the dog will
 * never run out of energy. The dog needs to pee.
 *
 * Note that this task picks a target. It does NOT move the dog. We want the
 * dog movement to take place outside of the tree, because the tree is not
 * updated every animaton frame. Separating movement from targeting makes the
 * animation smoother.
 */
public class WalkTask extends LeafTask<Dog> {
    /** The distance tolerance */
    private static final float DIST_TOLERANCE = 2f;

    /** A comparison vector so we do not corrupt the dog position */
    private final Vector2 compare = new Vector2();

	/**
	 * Initiates the run task.
	 *
	 * This method will pick a tree target to run to.
	 */
    @Override
    public void start () {
        super.start();
        Dog dog = getObject();
        dog.targetTree();
        dog.startWalking();
    }

	/**
	 * Executes one step of the run task
	 *
	 * This task will continue until the dog reaches its target
	 *
	 * @return The status of this task
	 */
    @Override
    public Task.Status execute () {
        Dog dog = getObject();
		Vector2 target = dog.getTarget();
        compare.set(dog.getPosition());
        if (compare.sub(target).len2() < DIST_TOLERANCE) {
            // Dog is close enough to target, success
            dog.brainLog("AT LAST - I am at the tree.");
            return Task.Status.SUCCEEDED;
        }

        // Dog is still moving towards target, running
        dog.brainLog("HURRY UP - Running to "+target.toString());
        return Task.Status.RUNNING;
    }

    /**
     * Completes this task
     *
     * This is called when the task is complete.
     */
	@Override
	public void end () {
        Dog dog = getObject();
        getObject().stopWalking();
	}


    @Override
    protected Task<Dog> copyTo (Task<Dog> task) {
        // No task attributes
        return task;
    }

}
