/*
 * Play.java
 *
 * This class represents running about and having fun. It is essentially the
 * default behavior in the behavior tree. It is adapted from the GDX AI example
 * written by davebaol.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
package edu.cornell.cis3152.behavior.tasks;


import com.badlogic.gdx.ai.GdxAI;
import com.badlogic.gdx.ai.btree.LeafTask;
import com.badlogic.gdx.ai.btree.Task;
import com.badlogic.gdx.math.Vector2;
import edu.cornell.cis3152.behavior.Dog;

/**
 * A task representing running about and having fun.
 *
 * This is the default behavior of the dog and generally succeeds. However, to
 * make things a little interesting, the dog has an energy level and the dog
 * this will fail when the dog is over-exerted. In the behavior tree, this causes
 * the dog to roll over to the {@link RestTask}.
 *
 * Note that this task picks a target. It does NOT move the dog. We want the
 * dog movement to take place outside of the tree, because the tree is not
 * updated every animaton frame. Separating movement from targeting makes the
 * animation smoother.
 */
public class PlayTask extends LeafTask<Dog> {
    /** The distance tolerance */
    private static final float DIST_TOLERANCE = 4f;

    /** A comparison vector so we do not corrupt the dog position */
    private final Vector2 compare = new Vector2();

	/**
	 * Initiates the play task.
	 *
	 * This method will pick a corner to run to.
	 */
	public void start () {
		Dog dog = getObject();
		dog.play();
		dog.brainLog("WOW - Lets play!");
		dog.targetCorner();
	}

	/**
	 * Executes one step of the play task
	 *
	 * This task will continue until the dog runs out of energy.
	 *
	 * @return The status of this task
	 */
	@Override
	public Status execute () {
		Dog dog = getObject();
		Vector2 target=dog.getTarget();
		dog.brainLog("PANT PANT - So fun");
        compare.set(dog.getPosition());
		if (target == null) {
			dog.targetCorner();
			return Status.RUNNING;
        } else if (dog.isExhausted()) {
            // Dog is out of energy, couldn't make it to target
            return Task.Status.FAILED;
		} else if (compare.sub(target).len2() < DIST_TOLERANCE) {
			// Dog is close enough to target, choose a new target
			dog.targetCorner();
			dog.brainLog("YEAH - I wonder what is over there!");
			return Status.RUNNING;
		} else{
            // Dog is still moving towards target, running
            dog.brainLog("SQUIRREL? - Running to target "+target.toString());
            return Task.Status.RUNNING;
        }
    }

	/**
	 * Copies the state of this task to another
	 *
	 * @param task  The task to copy into
	 */
	@Override
	protected Task<Dog> copyTo (Task<Dog> task) {
	    // No task attributes
		return task;
	}

		/**
     * Resets this task to start over
     */
	@Override
	public void reset() {
	    Dog dog = getObject();
        dog.setExhausted(false);
		super.reset();
	}
}
