/*
 * Care.java
 *
 * This class represents the need to pee. It is adapted from the GDX AI
 * example written by davebaol.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
package edu.cornell.cis3152.behavior.tasks;

import com.badlogic.gdx.ai.btree.LeafTask;
import com.badlogic.gdx.ai.btree.Task;
import com.badlogic.gdx.ai.btree.annotation.TaskAttribute;
import edu.cornell.cis3152.behavior.Dog;

/**
 * A task indicating a desire to pee.
 *
 * This task will FAIL if the dog needs to pee. That is how we communicate
 * to the behavior tree that the dog should go pee. It will force the tree
 * to switch to a new branch.
 */
public class CareTask extends LeafTask<Dog> {

    /** A task attribute for the pee probability (higher = more often) */
	@TaskAttribute(required = true)
	public float urgentProb = 0.1f;

	/**
	 * Executes one step of the care task
	 *
	 * The task is only executed once. Either it fully succeeds (no need to
	 * pee) or it fails.
	 *
	 * @return The status of this task
	 */
	@Override
	public Status execute () {
		Dog dog = getObject();
	    // If we already need to pee, we automatically fail
        if (dog.isUrgent()) {
    		return Status.FAILED;
        }

		if (Math.random() > urgentProb) {
			return Status.SUCCEEDED;
		}

        // We need to pee!
		dog.setUrgent(true);
		dog.brainLog("GASP - Something urgent :/");
		return Status.FAILED;
	}

	/**
	 * Copies the state of this task to another
	 *
	 * @param task  The task to copy into
	 */
	@Override
	protected Task<Dog> copyTo (Task<Dog> task) {
		CareTask care = (CareTask)task;
		care.urgentProb = urgentProb;

		return task;
	}

	/**
     * Resets this task to start over
     */
	@Override
	public void reset() {
        Dog dog = getObject();
		urgentProb = 0.1f;
		dog.setUrgent(false);
		super.reset();
	}

}
