/*
 * StatusListener.java
 *
 * This class shows how to attach a third party listener to a behavior tree.
 * It is used to update the label on the screen indicating the dog's action.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
 package edu.cornell.cis3152.behavior;

import com.badlogic.gdx.ai.btree.Task;
import com.badlogic.gdx.ai.btree.BehaviorTree;
import edu.cornell.cis3152.behavior.Dog;

/**
 * A listener for tracking a dogs action.
 *
 * This simple listener has a text string describing the dog's current action.
 * It is used to display this information on the screen.
 */
public class StatusListener implements BehaviorTree.Listener<Dog> {

    /** The current action */
    private String activity = "IDLE";

    /**
     * Creates a new status listener
     */
    public StatusListener() {
    }

    /**
     * Returns the current dog activity
     *
     * @return the current dog activity
     */
    public String getActivity() {
        return activity;
    }

    /**
     * Invoked when the task status is set.
     *
     * This does not necessarily mean that the status has changed. It simply
     * means that this task has been invoked by the tree.
     *
     * @param task      the task whose status has been set
     * @param previous  the task's previous status
     */
    @Override
    public void statusUpdated(Task<Dog> task, Task.Status previous) {
        String taskName = task.getClass().getSimpleName();

        // Only track high-level tasks like running, sleeping, playing, and peeing
        // Make sure we were not already in task
        if (previous != Task.Status.RUNNING) {
            if (taskName.contains("WalkTask")) {
                activity = "Walking";
            } else if (taskName.contains("RestTask") || taskName.contains("WaitTask")) {
                activity = "Sleeping";
            } else if (taskName.contains("PlayTask")) {
                activity = "Playing";
            } else if (taskName.contains("BarkTask")) {
                activity = "Barking";
            } else if (taskName.contains("MarkTask")) {
                activity = "Peeing";
            }
        }
    }

    /**
     * Invoked when a child task is added to the children of a parent task.
     *
     * @param parent    the parent task of the newly added child
     * @param index     the index where the child has been added */
    @Override
    public void childAdded(Task<Dog> parent, int index) {
        // This is NEVER called in this demo (the tree never changes)
        Task<Dog> childTask = parent.getChild(index);
        System.out.println("Child task added: " + childTask.getClass().getSimpleName());
    }
}

