/*
 * Entity.java
 *
 * This is a class used to polymorphically represent objects in our demo. Even
 * though not all objects are animated, it is designed around sprites. We just
 * assume that static images are sprites that are not animated
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
package edu.cornell.cis3152.behavior;


import com.badlogic.gdx.graphics.Texture;
import com.badlogic.gdx.graphics.g2d.Batch;
import com.badlogic.gdx.graphics.g2d.TextureRegion;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.utils.Disposable;
import edu.cornell.gdiac.graphics.SpriteBatch;
import edu.cornell.gdiac.graphics.SpriteSheet;
import java.util.Comparator;


/**
 * The base class for game object in this demo.
 *
 * Our game objects will be drawn assuming we are using a 3/4 perspective.
 * As such, the drawing origin the middle bottom of the image. We also order
 * these objects by their y-value, so that they can be sorted before drawing.
 * This prevents artifacts such as one object hanging in the air above another.
 */
public class Entity implements Comparable<Entity> {
    /** The entity position */
    protected final Vector2 position = new Vector2();
    /** The entity image/sprite sheet */
    protected SpriteSheet spriteSheet;
    /** Whether the entity is facing left (if not the image is flipped) */
    protected boolean facingLeft;
    /** The number of frames until we increment the sprite frame (-1 for still image) */
    protected int frameDelay = -1;
    /** The current frame counter, used to increment the sprite frame */
    protected int frameCounter = 0;

    /** A static comparator for comparing entities (used in the draw sort) */
    public static Comparator<Entity>  Comparison = new Comparator<Entity>() {

        /**
         * Compares the two arguments for order.
         *
         * This method returns a negative integer, zero, or a positive integer
         * as the first argument is less than, equal to, or greater than the
         * second.
         *
         * @param o1    the first object to be compared
         * @param o2    the second object to be compared
         */
        @Override
        public int compare(Entity o1, Entity o2) {
            return o1.compareTo(o2);
        }
    };

    /**
     * Creates an entity positioned at the origin
     */
    public Entity() {
        this(0,0);
    }

    /**
     * Creates an entity positioned at the given location
     *
     * @param x The entity x-coordinate
     * @param y The entity y-coordinate
     */
    public Entity(float x, float y) {
        position.set(x, y);
        facingLeft = true;
    }

    /**
     * Returns the position of this entity.
     *
     * The object returned is a reference to the position. Changing this
     * object will modify the entity position.
     *
     * @return the position of this entity.
     */
    public Vector2 getPosition() {
        return position;
    }

    /**
     * Sets the position of this entity
     *
     * @param x The entity x-coordinate
     * @param y The entity y-coordinate
     */
    public void setPosition(float x, float y) {
        position.set(x, y);
    }


    /**
     * Returns the sprite sheet for this entity.
     *
     * It is possible for this value to be null. In that case, nothing is
     * drawn.
     *
     * @return the sprite sheet for this entity.
     */
    public SpriteSheet getSpriteSheet() {
        return spriteSheet;
    }

    /**
     * Sets the sprite sheet for this entity.
     *
     * If this value is null, this method will allocate a blank sprite sheet.
     *
     * @param sheet The new sprite sheet for this entity
     */
    public void setSpriteSheet(SpriteSheet sheet) {
        if (spriteSheet == null) {
            spriteSheet = new SpriteSheet();
        }
        spriteSheet.set( sheet );
    }

    /**
     * Sets the texture for this entity.
     *
     * This method sets the sprite sheet to have a single animation frame
     * encompasing the given texture.
     *
     * @param sheet The new texture for this entity
     */
    public void setTexture(Texture texture) {
        if (spriteSheet == null) {
            spriteSheet = new SpriteSheet();
        }
        if (texture != null) {
            spriteSheet.set( texture, 1, 1 );
        } else {
            spriteSheet.set( null );
        }
    }

    /**
     * Draws this entity to the given sprite batch
     *
     * If this entity is animated, this method will increment the animation
     * frame.
     *
     * @param batch The sprite batch to draw to
     */
    public void draw(SpriteBatch batch) {
        if (spriteSheet == null) {
            return;
        }

        if (facingLeft == spriteSheet.isFlipX()) {
            spriteSheet.flip(true, false);
        }

        // Draw at bottom center
        float x = position.x - spriteSheet.getRegionWidth()/2f;
        batch.draw(spriteSheet, x, position.y);

        // If this is animated, increment the frame
        if (frameDelay > 0) {
            progressFrame();
        }
    }

    /**
     * Progresses the animation frame one step
     *
     * This method uses the frameCounter and frameDelay to determine when to
     * advance the sprite frame.
     */
    private void progressFrame() {
        if (frameDelay == 0 || frameCounter % frameDelay == 0) {
            if (spriteSheet.getFrame() < spriteSheet.getSize() - 1) {
                spriteSheet.setFrame( spriteSheet.getFrame() + 1 );
            } else {
                spriteSheet.setFrame(0);
            }
        }
        frameCounter++;
    }

    /**
     * Compares the arguments to this object for order.
     *
     * This method returns a negative integer, zero, or a positive integer
     * as the this object is less than, equal to, or greater than the
     * argument.
     *
     * @param o    the object to be compared
     */
    @Override
    public int compareTo(Entity o) {
        // Sort on y, then on x
        if (position.y > o.position.y) {
            return -1;
        } else if (position.y < o.position.y) {
            return 1;
        } else if (position.x > o.position.x) {
            return -1;
        } else if (position.x < o.position.x) {
            return 1;
        }
        return 0;
    }
}
