/**
 * Dog.java
 *
 * This class is the one being controlled by the behavior tree. It is adapted
 * from the GDX AI example written by davebaol.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
package edu.cornell.cis3152.behavior;
import com.badlogic.gdx.ai.GdxAI;
import com.badlogic.gdx.math.MathUtils;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.utils.JsonValue;
import edu.cornell.gdiac.assets.AssetDirectory;
import edu.cornell.gdiac.graphics.SpriteSheet;

import java.util.ArrayList;
import java.util.Random;

/**
 * The class representing our AI controlled dog.
 *
 * The AI is not embedded into this class. It is handled externally by the
 * individual behavior tree tasks. Instead, this class just consists of getter
 * and setter styled hooks for controlling the dog.
 */
public class Dog extends Entity {
    /** The name of the dog */
	public String name;
	/** The name of the dog's brain (for displaying to the console) */
	public String brainLog;

	/** The amount of energy the dog has in seconds */
	private float maxEnergy;
    /** The current energy the dog has in seconds */
    private float energy;
	/** The walking speed of the dog in pixels */
	private float speed;
	/** Whether the dog is currently exhausted */
	private boolean exhausted;
	/** Whether the dog "needs to go" */
    private boolean urgent = false;

    /** The current movement target for the dog */
	private Vector2 target;
	/** The corners of the screen (target candidates) */
    private ArrayList<Vector2> corners;
	/** The tree locations (target candidates) */
	private ArrayList<Vector2> trees;
    /** The sprite for walking */
	private SpriteSheet walkSprite;
    /** The sprite for sleeping */
	private SpriteSheet sleepSprite;
    /** The sprite for barking */
	private SpriteSheet barkSprite;
    /** The sprite for marking territory */
	private SpriteSheet markSprite;
	/** The sprite for idle animations */
    private SpriteSheet idleSprite;

    /** An internal comparison vector */
    private final Vector2 comparison = new Vector2();

    /**
     * Creates a dog from the given data and asset directory.
     *
     * The asset directory should represent the ENTIRE level. In addition to
     * is own settings, the dog needs to know the size of the screen and the
     * locations of the trees. All of this should be in the given JsonValue.
     *
     * The asset directory is used to specify the dog sprites.
     *
     * @param data      The level settings
     * @param assets    The game assets
     */
	public Dog (JsonValue data, AssetDirectory assets) {
        JsonValue dog = data.get("dog");

        name  = dog.getString("name", "Fido");
        speed = dog.getFloat("speed", 30f);
        energy = dog.getFloat("energy", 15f);;
		maxEnergy  = energy;
        frameDelay = dog.getInt("frame delay", 12);
		brainLog = name + " brain";

        JsonValue pos = dog.get("position");
        position.x = pos.getFloat(0);
        position.y = pos.getFloat(1);

        walkSprite = assets.getEntry("dog-walk.sprite", SpriteSheet.class);
        sleepSprite = assets.getEntry("dog-rest.sprite", SpriteSheet.class);
        barkSprite = assets.getEntry("dog-bark.sprite", SpriteSheet.class);
        idleSprite = assets.getEntry("dog-idle.sprite", SpriteSheet.class);
        markSprite = assets.getEntry("dog-mark.sprite", SpriteSheet.class);
        spriteSheet = idleSprite;

        // Get the corners
        JsonValue size = data.get("size");
        float left = walkSprite.getRegionWidth()/2;
        float bot = 0;
        float right = size.getFloat(0)-left;
        float top = size.getFloat(1)-walkSprite.getRegionHeight();

        corners = new ArrayList<>();
        corners.add(new Vector2(left,bot));
        corners.add(new Vector2(left,top));
        corners.add(new Vector2(right, bot));
        corners.add(new Vector2(right, top));

        // Get the trees
        JsonValue treePos = data.get("trees");
        trees = new ArrayList<>();
        for(int ii=0; ii<treePos.size; ii++) {
            pos = treePos.get(ii);
            trees.add( new Vector2(pos.getFloat(0),pos.getFloat(1)) );
        }
    }

    /**
     * Returns the current movement target
     *
     * @return the current movement target
     */
    public Vector2 getTarget() {
        return this.target;
	}


    /**
     * Chooses a tree as the current target
     */
    public boolean targetTree() {
		if (trees == null || trees.isEmpty()) return false;

		Vector2 currentPosition = new Vector2(position.x, position.y);
		Vector2 nearestTarget = null;
		float nearestDistance = Float.MAX_VALUE;  // Value to beat
		for (Vector2 tree : trees) {
			float distance = currentPosition.dst(tree);
			if (distance < nearestDistance && !tree.equals(target)) {
				nearestDistance = distance;
				nearestTarget = tree;
			}
		}
		// If we found a valid nearest target, update the target and facing direction
		if (nearestTarget != null) {
			target = nearestTarget;
			facingLeft = target.x - position.x < 0;// Update facing direction
			return true;
		}
		return false;
	}

	/**
	 * Chooses a corner of the screen as the current target
	 */
	public boolean targetCorner() {
		Random rand = new Random();
		int newTargetIndex = rand.nextInt(corners.size());
		if (corners.get(newTargetIndex).equals(target)) {
            newTargetIndex = (newTargetIndex + 1) % corners.size();
        }
		target = corners.get(newTargetIndex);
		facingLeft = target.x - position.x < 0;
		return true;
	}

    /**
     * Moves the dog towards the target
     *
     * This movement should be applied every animation frame. Therefore this
     * method is not invoked in the behavior tasks.  The tasks simply pick a
     * target for this action.
     *
     * Calling this method expends "energy". After a while, the dog will be
     * exhausted. But an exhausted dog is still allowed to move (such as when
     * the dog really needs to go).
     *
     * @param delta The number of seconds to move the dog
     */
	public void moveToTarget(float delta) {
        // We do not move if there is no target
		if (target == null) {
            return;
        }

		// Movement vector to target scaled to move speed
        comparison.set(target);
        comparison.sub(position);

        // We are there
        if (comparison.len2() < 1.0f) {
            return;
        }

        comparison.nor().scl(speed*delta);
		position.add(comparison);

        // Keep track of this exertion
        energy -= delta;
        if (energy <= 0) {
            energy = 0;
            exhausted = true;
        }
	}

    /**
     * Returns the dog's current energy.
     *
     * This is the number of seconds the dog has left to act. When exceeded, it
     * will go to sleep.
     *
     * @return the movement budget for this dog
     */
	public float getEnergy() {
		return energy;
	}

    /**
     * Returns true if this dog is exhausted
     *
     * An exhausted dog cannot run anymore. The behavior tree has to pick a
     * new task.
     *
     * @return true if this dog is exhausted
     */
	public boolean isExhausted() {
		return exhausted;
	}

    /**
     * Sets whether this dog is exhausted
     *
     * An exhausted dog cannot run anymore. The behavior tree has to pick a
     * new task. Setting this to false will reset the dog's energy.
     *
     * @param exhausted     Whether this dog is exhausted
     */
	public void setExhausted(boolean exhausted){
        this.exhausted=exhausted;
        if (!exhausted) {
            energy = maxEnergy;
        }
	}

	/**
	 * Returns true if the dog needs to go
	 *
	 * @return true if the dog needs to go
	 */
	public boolean isUrgent () {
		return urgent;
	}

	/**
	 * Sets whether the dog needs to go
	 *
	 * @param urgent    Whether the dog needs to go
	 */
	public void setUrgent (boolean urgent) {
		this.urgent = urgent;
	}

	// TASK METHODS

    /**
     * Performs a bark task
     *
     * This method activates the bark animation and outputs a bark noise to
     * the console
     */
	public void bark () {
        if (spriteSheet != barkSprite) {
            spriteSheet = barkSprite;
        }
        target = null;
		if (MathUtils.randomBoolean()) {
            log( "Arf arf" );
        } else {
            log( "Woof" );
        }
	}

   /**
     * Performs a sleep task
     *
     * This method activates the sleep animation and clears the exhaustion
     * setting.
     */
    public void sleep() {
		if (spriteSheet != sleepSprite) {
            spriteSheet = sleepSprite;
        }
        target = null;
		setExhausted(false);
	}

	/**
	 * Performs a play task
	 *
	 * This method activates the walking animation with no commentary.
	 */
	public void play() {
		if (spriteSheet != walkSprite) {
            spriteSheet = walkSprite;
        }
	}

	/**
	 * Initiates a walking task
	 *
	 * This method activates the walking animation with commentary on the
	 * dog's actions.
	 */
	public void startWalking() {
		if (spriteSheet != walkSprite) {
            spriteSheet = walkSprite;
        }
		log("I found a tree. Going there!");
	}

    /**
     * Completes a walking task
     *
     * We do not change the animation until we know the new task. This makes
     * sure the dog is in front of the tree.
     */
	public void stopWalking() {
        // Push our y below the target so we sort in front
        if (target != null) {
            if (position.y >= target.y) {
                position.y = target.y-1;
            }
        }
		log("This tree smells good :)");
	}

    /**
     * Performs a marking task
     *
     * This method activates the peeing animation. This method returns false if
     * the dog is still peeing and true if it has finished.
     *
     * @param turn  The number of (AI) turns spent peeing
     *
     * @return true if the dog has finished
     */
	public boolean markTree (int turn) {
		if (spriteSheet != markSprite) {
            spriteSheet = markSprite;
        }
		if (turn == 0) {
			log("Swoosh....");
			return false;
		}
		log("I'm ok now :)");
		return true;
	}

	/**
	 * Logs the given message to the console
	 *
	 * This log is annotated by the dog's name. It is for actions carried out
	 * by this class.
	 *
	 * @param msg   The message to log
	 */
	public void log (String msg) {
		GdxAI.getLogger().info(name, msg);
	}

	/**
	 * Logs the given message to the console
	 *
	 * This log is annotated by the dog's brain. It is for actions carried out
	 * by the dog's behavior tree.
	 *
	 * @param msg   The message to log
	 */
	public void brainLog (String msg) {
		GdxAI.getLogger().info(brainLog, msg);
	}

}
