/*
 * BehaviorScene.java
 *
 * This scene is a reworking of the Behavior Tree demo included in GDX AI. It
 * makes too major changes.  First of all, it removes the scene2 Stage
 * infrastructure that obfuscated the behavior loop. Second, it combines the
 * behavior with animations, showing how several actions may span multiple
 * timesteps.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
 package edu.cornell.cis3152.behavior;

import com.badlogic.gdx.*;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.Color;
import com.badlogic.gdx.graphics.g2d.BitmapFont;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.utils.JsonValue;
import com.badlogic.gdx.utils.ScreenUtils;
import edu.cornell.gdiac.assets.*;
import edu.cornell.gdiac.graphics.*;
import edu.cornell.gdiac.util.ScreenListener;

import java.util.ArrayList;
import java.util.Set;


/**
 * Gameplay controller for the behavioral demo.
 *
 * This class is runs and displays the a demo of a dog running about the lawn
 * (with frequent bursts of urgency). It is primarily responsible for
 * initializing the scene and rendering it to the screen. The real work of the
 * game is handled by {@link BehaviorLoop}.
 */
public class BehaviorScene implements Screen {
    /** The scene width */
    private float width;
    /** The scene height */
    private float height;

    /** The directory with the image file */
    private AssetDirectory directory;
    /** The drawing camera for this scene (VIEW CLASS) */
    private OrthographicCamera camera;
    /** The 2d graphics pipeline (VIEW CLASS) */
    private SpriteBatch batch;

    /** Listener that will update the player mode when we are done */
    private ScreenListener screenlistener;
    /** A listener to monitor the behavior tree */
    private StatusListener statusListener;

    /** The background image */
    private Texture backgroundTexture;
    /** The text displaying the current activity */
    private TextLayout taskLabel;
    /** List of all the objects in the game */
    private ArrayList<Entity> entities = new ArrayList<>();
    /** A reference to the dog (which is in the entity list) */
    private Dog dog;

    /** A factory to create a behavior tree (as we do not have a loader) */
    private BehaviorFactory factory;
    /** The gameplay loop for the dogs behavior tree */
    private BehaviorLoop<Dog> behavior;

    /**
     * Creates a new demo from the given assets and sprite batch
     *
     * @param assets	The asset directory containing all the loaded assets
     * @param batch     The sprite batch to draw to
     */
    public BehaviorScene(AssetDirectory assets, SpriteBatch batch) {
        this.directory = assets;
        this.batch = batch;

        // We don't really have a good asset loader for this. Load directly.
        factory = new BehaviorFactory(Gdx.files.internal("behaviors/dog.tree"));

        JsonValue level = directory.getEntry( "level", JsonValue.class );
        JsonValue size = level.get("size");
        resize( size.getInt( 0 ),size.getInt( 1 ) );

        backgroundTexture = directory.getEntry( "background", Texture.class );


        spawnTrees(level.get( "trees" ));


        dog = new Dog(level, directory);
        entities.add(dog);

        statusListener = new StatusListener();
        behavior = factory.createActor(dog, statusListener);


        taskLabel = new TextLayout();
        taskLabel.setFont( directory.getEntry( "daydream", BitmapFont.class ));
        taskLabel.setColor(Color.WHITE);
        taskLabel.setAlignment( TextAlign.bottomLeft );
        taskLabel.setText( "Current Status: None" );

    }

    /**
     * Creates the trees for the dog to interact with
     *
     * @param trees The JSON entry specifying the tree locations
     */
    private void spawnTrees(JsonValue trees) {
        Texture texture = directory.getEntry( "tree", Texture.class );
        for (int ii = 0; ii < trees.size; ii++) {
            JsonValue pos = trees.get(ii);
            float xOffset = pos.getFloat( 0 );
            float yOffset = pos.getFloat( 1 );

            Entity tree = new Entity(xOffset, yOffset);
            tree.setTexture( texture );
            entities.add(tree);
        }
    }

	/**
	 * Steps the core gameplay loop of this world.
	 *
	 * As you can see, this method primarily delegates to the behavior loop.
	 * However, we do move the dog separately. Since the behavior tree only
	 * updates every couple of frames, we need to animate the dog separately.
	 *
	 * @param delta Number of seconds since last animation frame
	 */
    public void update(float delta) {
        behavior.update(delta);
        dog.moveToTarget( delta );
    }

    /**
     * Draws the objects to the game.
     *
     * This method is similar to other demos with one major difference. This
     * game uses a 3/4 perspective for the assets. To get them to tile correctly,
     * we have to sort them with respect to the y-axis before drawing them. As
     * the dog can move up and down, we have to do this every frame.
     */
    public void draw() {
        ScreenUtils.clear(Color.BLACK);

        batch.begin(camera);
        batch.draw(backgroundTexture, 0, 0, width, height);

        // Sort the 3/4 perspective
        entities.sort( Entity.Comparison );
        for(Entity entity : entities) {
            entity.draw( batch );
        }

        String currentActivity = statusListener.getActivity();

        taskLabel.setText("Current Status: " + currentActivity);
        taskLabel.layout();
        batch.drawText( taskLabel, 10, 10 );

        batch.end();
    }

    /**
     * Sets the ScreenListener for this scene
     *
     * The ScreenListener will respond to requests to quit.
     */
    public void setScreenListener(ScreenListener listener) {
        screenlistener = listener;
    }

	/**
	 * Called when the Screen is resized.
	 *
	 * This can happen at any point during a non-paused state but will never
	 * happen before a call to show().
	 *
	 * @param width  The new width in pixels
	 * @param height The new height in pixels
	 */
    @Override
    public void resize(int width, int height) {
        this.width = width;
        this.height = height;

        if (camera == null) {
            camera = new OrthographicCamera();
        }
        camera.setToOrtho(false, width, height);
    }

	/**
	 * Called when the Screen should render itself.
	 *
	 * We defer to the other methods update() and draw(). However, it is VERY
	 * important that we only quit AFTER a draw.
	 *
	 * @param delta Number of seconds since last animation frame
	 */
    @Override
    public void render(float delta) {
        update(delta);
        if (batch != null) {
            draw();
        }
    }

	/**
	 * Dispose of all (non-static) resources allocated to this mode.
	 */
    @Override
    public void dispose() {
    }

	/**
	 * Called when the Screen is paused.
	 *
	 * This is usually when it's not active or visible on screen. An Application
	 * is also paused before it is destroyed.
	 */
    @Override
    public void pause() {
    }

	/**
	 * Called when the Screen is resumed from a paused state.
	 *
	 * This is usually when it regains focus.
	 */
    @Override
    public void resume() {
    }


	/**
	 * Called when this screen is no longer the current screen for a Game.
	 */
    @Override
    public void hide() {
    }

	/**
	 * Called when this screen becomes the current screen for a Game.
	 */
    @Override
    public void show() {
    }

}
