/*
 * BehaviorLoop.java
 *
 * This class is a game loop for running a (single) behavior tree. In the GDX
 * AI demo, they used the Stage API to do the same thing. This makes the loop
 * more explicit. In practice, you would not have a loop for just one behavior
 * tree. Instead, you would have multiple trees being processed simultaneously.
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
package edu.cornell.cis3152.behavior;

import com.badlogic.gdx.ai.GdxAI;
import com.badlogic.gdx.ai.btree.BehaviorTree;

/**
 * A class to drive a single behavior tree
 *
 * The behavior tree does not run at animation frame rate (no one thinks that
 * fast). Instead, it has a run delay and is only invoked periodically. In
 * practice, we should make this delay user-definable.
 */
public class BehaviorLoop<E> {
    /** The behavior tree to control */
    private BehaviorTree<E> tree;
    /** The current AI step */
    private int step;
    /** The delay between AI calls (in seconds) */
    private float runDelay;
    /** The time (in seconds) since the last call to the behavior tree */
    private float runAmount;

    /**
     * Creates a gameplay loop for the given behavior tree
     *
     * @param tree  The behavior tree to manage
     */
    public BehaviorLoop(BehaviorTree<E> tree) {
        this.tree = tree;
        this.step = 0;

        // Over have a second between steps
        runDelay = 0.6f;
        runAmount = 0;
    }

    /**
     * Updates the behavior loop, moving all tasks forward
     *
     * Behavior works on a delay. We only advance the behavior tree after
     * runDelay seconds have passed
     *
     * @param delta The amount of time to advance the loop
     */
    public void update(float delta) {
        // We need to communicate to the AI subsystem how much time has passed
        GdxAI.getTimepiece().update(delta);
        runAmount += delta;
        if (runAmount > runDelay) {
            tree.step();
            step++;
            System.out.println("Step " + step);
            runAmount = 0;
        }
    }

    /**
     * Returns the behavior tree for this gameplay loop
     *
     * @return the behavior tree for this gameplay loop
     */
    public BehaviorTree<E> getBehaviorTree() {
        return tree;
    }

}
