/*
 * BehaviorFactory.java
 *
 * This class is used to load a behavior tree file and attach a behavior loop
 * to an object. We need this class because there is no asset loader for
 * behavior trees. The generic features of behavior trees makes it difficult
 * to use them in an asset directory (at best we could load the source string)
 *
 * @author: David Kim and James Liu
 * @version: 3/10/2025
 */
 package edu.cornell.cis3152.behavior;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.ai.btree.BehaviorTree;
import com.badlogic.gdx.ai.btree.utils.BehaviorTreeParser;
import com.badlogic.gdx.files.FileHandle;

import java.io.IOException;
import java.io.Reader;

/**
 * A factory to generate a gameplay loop from a behavior tree
 *
 * This factory can be used to apply the same behavior tree to multiple objects.
 * Each blackboard object will get its own gameplay loop
 */
public class BehaviorFactory<E> {
    /** The behavior tree "source code" */
    String source;

    /**
     * Create a behavior tree factory from the source code
     *
     * @param code  The behavior tree source code
     */
    public BehaviorFactory(String code) {
        source = code;
    }

    /**
     * Create a behavior tree factory from the given file
     *
     * @param file  The file with the behavior tree source code
     */
    public BehaviorFactory(FileHandle file) {
        try {
            Reader reader = file.reader();
            StringBuilder builder = new StringBuilder();
            int c = reader.read();
            while (c != -1) {
                builder.append((char) c);
                c = reader.read();
            }
            source = builder.toString();
        } catch (IOException e) {

        }
    }

    /**
     * Returns a gameplay loop for the given blackboard object
     *
     * @param object    The blackboard object
     */
    public BehaviorLoop<E> createActor(E object) {
        BehaviorTreeParser<E> parser = new BehaviorTreeParser<E>(BehaviorTreeParser.DEBUG_HIGH);
        BehaviorTree<E> tree = parser.parse(source, object);
        BehaviorLoop<E> result = new BehaviorLoop<E>(tree);
        return result;
    }

    /**
     * Returns a gameplay loop for the given blackboard object
     *
     * This method will also attach the given external listener to the behavior
     * tree.
     *
     * @param object    The blackboard object
     * @param listener  The tree listener
     */
    public BehaviorLoop<E> createActor(E object, BehaviorTree.Listener<E> listener) {
        BehaviorTreeParser<E> parser = new BehaviorTreeParser<E>(BehaviorTreeParser.DEBUG_HIGH);
        BehaviorTree<E> tree = parser.parse(source, object);
        tree.addListener(listener);
        BehaviorLoop<E> result = new BehaviorLoop<E>(tree);
        return result;
    }

}
