/*
 * GDXRoot.java
 *
 * This is the primary class file for running the game.  It is the "static main" of
 * LibGDX; it must extend ApplicationAdapter to work properly.
 *
 * In this demo we did not break the game into different modes.  Everything is done
 * in the root.  That is because there are no assets to load, and all this application
 * does is draw geometric primitives.
 *
 * @author Crystal Jin
 * @date   12/5/2022
 */
package edu.cornell.gdiac.shapes;

import com.badlogic.gdx.*;
import com.badlogic.gdx.graphics.g2d.*;
import com.badlogic.gdx.graphics.*;
import edu.cornell.gdiac.math.*;

import java.util.ArrayList;

/**
 * Root class for the graphics demo.
 *
 *
 * This class is technically not the ROOT CLASS. Each platform has another class above
 * this (e.g. PC games use DesktopLauncher) which serves as the true root.  However,
 * those classes are unique to each platform, while this class is the same across all
 * plaforms. In addition, this functions as the root class all intents and purposes,
 * and you would draw it as a root class in an architecture specification.
 */
public class GDXRoot extends ApplicationAdapter implements InputProcessor{
    private State state= State.PolyFactory;

    private BitmapFont displayFont;
    private PolygonSpriteBatch batch;
    private TextureRegion region;

    private PolyFactory polyFactory;
    private PolyTriangulator triangulator;
    private PathExtruder extruder;
    private PathFactory pathFactory;
    private SplinePather pather;

    private Poly2 temp;
    private Path2 temp1;
    private Spline2 temp2;
    private int PATH_WIDTH = 5;

    private enum State {
        PolyFactory,
        PathFactory,
        Triangulator,
        Extruder,
        PathSmoother,
        Spline
    }

    //State 0 or 2
    private PolygonRegion rect;
    private PolygonRegion circle;
    private PolygonRegion triangle;
    private PolygonRegion ngon6;
    private PolygonRegion ngon8;
    private PolygonRegion ellipse;
    private PolygonRegion arc;
    private PolygonRegion rRect;
    /** Capsule with w<h, in full shape*/
    private PolygonRegion capsule1;
    /** Capsule with w>h, in full shape*/
    private PolygonRegion capsule2;
    /** Capsule with w<h, in half shape*/
    private PolygonRegion capsule3;
    /** Capsule with w<h, in half_reversed shape*/
    private PolygonRegion capsule4;
    /** Capsule with w>h, in half shape*/
    private PolygonRegion capsule5;
    /** Capsule with w>h, in half_reversed shape*/
    private PolygonRegion capsule6;

    // State 3
    private PolygonRegion polyRegion;

    //State 1
    private PolygonRegion pathRegion1;
    private PolygonRegion pathRegion2;
    private PolygonRegion pathRegion3;
    private float CIRCLE[] = {    0,  200,  120,  200,
            200,  120,  200,    0,  200, -120,
            120, -200,    0, -200, -120, -200,
            -200, -120, -200,    0, -200,  120,
            -120,  200,    0,  200};
    private float STAR[] = {     0,    50,  10.75f,    17,   47,     17,
            17.88f, -4.88f,   29.5f, -40.5f,    0, -18.33f,
            -29.5f, -40.5f, -17.88f, -4.88f,  -47,     17,
            -10.75f,    17};
    private int LINE_WIDTH = 10;

    //State 4
    private PathSmoother smoother;
    private ArrayList<Float> points;
    private PolygonRegion smoothPoly;
    private boolean active;
    private int inputNum;
    private int outputNum;

    //State 5
    private PolygonRegion splinePoly;
    private PolygonRegion pathPoly;

    /**
     * Called when the Application is first created.
     *
     * This is method loads the image and defines the (initial) vertex buffer
     */
    @Override
    public void create () {
        batch = new PolygonSpriteBatch();
        region = new TextureRegion(new Texture("white.png"));
        displayFont = new BitmapFont();

        polyFactory = new PolyFactory();
        triangulator = new PolyTriangulator();
        pathFactory = new PathFactory();
        extruder = new PathExtruder();
        smoother = new PathSmoother();

        // Prepare to receive input events
        Gdx.input.setInputProcessor(this);

        initState(state);
    }

    private void handlePath(){
        extruder.set(temp1);
        extruder.calculate(PATH_WIDTH);
        temp = extruder.getPolygon();
    }

    private void initState(State state){
        switch (state){
            case PolyFactory:
                temp = polyFactory.makeRect(50, 25, 75, 50);
                rect = temp.makePolyRegion(region);

                temp = polyFactory.makeCircle(225, 50,40);
                circle = temp.makePolyRegion(region);

                temp = polyFactory.makeTriangle(300,25,400,25,375,100);
                triangle = temp.makePolyRegion(region);

                temp = polyFactory.makeEllipse(500,50,100,50);
                ellipse = temp.makePolyRegion(region);

                temp = polyFactory.makeNgon(75,175,50,6);
                ngon6 = temp.makePolyRegion(region);

                temp = polyFactory.makeNgon(200,175,50,8);
                ngon8 = temp.makePolyRegion(region);

                temp = polyFactory.makeArc(350,175,100,30,270);
                arc = temp.makePolyRegion(region);

                temp = polyFactory.makeRoundedRect(450, 125,100,75,(float)(Math.PI)*5);
                rRect = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(50,275,75,125);
                capsule1 = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(Poly2.Capsule.HALF,175,275,50,75);
                capsule3 = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(Poly2.Capsule.HALF_REVERSE,175,325,50,75);
                capsule4 = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(275,300,100,75);
                capsule2 = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(Poly2.Capsule.HALF,425,300,100,75);
                capsule5 = temp.makePolyRegion(region);

                temp = polyFactory.makeCapsule(Poly2.Capsule.HALF_REVERSE,475,300,100,75);
                capsule6 = temp.makePolyRegion(region);
                break;
            case Extruder:
                extruder.set(STAR,true);
                extruder.calculate(LINE_WIDTH);
                temp = extruder.getPolygon();
                pathRegion2 = temp.makePolyRegion(region);
                extruder.setJoint(Poly2.Joint.MITRE);
                extruder.calculate(LINE_WIDTH);
                temp = extruder.getPolygon();
                pathRegion1 = temp.makePolyRegion(region);
                extruder.setJoint(Poly2.Joint.ROUND);
                extruder.calculate(LINE_WIDTH);
                temp = extruder.getPolygon();
                pathRegion3 = temp.makePolyRegion(region);
                break;
            case PathFactory:
                temp1 = pathFactory.makeRect(50, 25, 75, 50);
                handlePath();
                rect = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCircle(225, 50,40);
                handlePath();
                circle = temp.makePolyRegion(region);

                temp1 = pathFactory.makeTriangle(300,25,400,25,375,100);
                handlePath();
                triangle = temp.makePolyRegion(region);

                temp1 = pathFactory.makeEllipse(500,50,100,50);
                handlePath();
                ellipse = temp.makePolyRegion(region);

                temp1 = pathFactory.makeNgon(75,175,50,6);
                handlePath();
                ngon6 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeNgon(200,175,50,8);
                handlePath();
                ngon8 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeArc(350,175,100,30,270, true);
                handlePath();
                arc = temp.makePolyRegion(region);

                temp1 = pathFactory.makeRoundedRect(450, 125,100,75,(float)(Math.PI)*5);
                handlePath();
                rRect = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(50,275,75,125);
                handlePath();
                capsule1 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(Poly2.Capsule.HALF,175,275,50,75);
                handlePath();
                capsule3 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(Poly2.Capsule.HALF_REVERSE,175,325,50,75);
                handlePath();
                capsule4 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(275,300,100,75);
                handlePath();
                capsule2 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(Poly2.Capsule.HALF,425,300,100,75);
                handlePath();
                capsule5 = temp.makePolyRegion(region);

                temp1 = pathFactory.makeCapsule(Poly2.Capsule.HALF_REVERSE,475,300,100,75);
                handlePath();
                capsule6 = temp.makePolyRegion(region);
                break;
            case Triangulator:
                // Convert circle spline to path
                pather = new SplinePather();
                temp2 = new Spline2(CIRCLE);
                temp2.setClosed( true );
                pather.set(temp2);
                pather.calculate();
                temp1 = pather.getPath();
                temp1.reverse();

                // Triangulate with hole
                triangulator.clear();
                triangulator.set(temp1);
                temp1 = new Path2(STAR);
                temp1.scl( 2.0f );
                triangulator.addHole( temp1 );

                triangulator.calculate();
                temp = triangulator.getPolygon();

                // Get object to draw
                polyRegion = temp.makePolyRegion( region );
                break;
            case PathSmoother:
                points = new ArrayList<>();
                smoothPoly = null;
                inputNum = 0;
                outputNum = 0;
                break;
            case Spline:
                points = new ArrayList<>();
                splinePoly = null;
                pathPoly = null;
                break;
            default:
                break;
        }
    }

    /**
     * Resets the polygon to match the current global state.
     *
     * The polygon created will be centered in the screen with size number of
     * edges.  Any distortions applied by the mouse will be removed.  If the texture
     * is turned off, the vertices will be colored using an HSV color-wheel.
     */
    public void reset() {

    }

    /**
     * Renders the polygon to the screen
     *
     * We do not need an update() call for this class, that is handled by the
     * InputProcessor methods.
     */
    @Override
    public void render () {

        Gdx.gl.glClearColor(0.7f, 0.7f, 0.7f, 0.75f);
        Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);

        batch.begin();
        drawText("Current state: "+state.name(), 25,Gdx.graphics.getHeight()-25);
        drawText("Press Left/Right key to switch state", 25,Gdx.graphics.getHeight()-50);

        if (state == State.PolyFactory || state == State.PathFactory){
            batch.setColor(1f, 0f, 0f, 0.75f);
            batch.draw(rect, 0,0);

            batch.setColor(0f, 1f, 0f, 0.75f);
            batch.draw(circle, 0,0);

            batch.setColor(0,0,1,0.75f);
            batch.draw(triangle,0,0);

            batch.setColor(0.5f,0.5f,0,0.75f);
            batch.draw(ngon6,0,0);

            batch.setColor(0.5f,0,0.5f,0.75f);
            batch.draw(ngon8,0,0);

            batch.setColor(0,0.5f,0.5f,0.75f);
            batch.draw(ellipse,0,0);

            batch.setColor(0.5f,0.25f,0.5f,0.75f);
            batch.draw(arc,0,0);

            batch.setColor(0.25f,0.5f,0.5f,0.75f);
            batch.draw(rRect,0,0);

            batch.setColor(0.5f,0.5f,0.25f,0.75f);
            batch.draw(capsule1,0,0);

            batch.setColor(0.25f,0.25f,0.75f,0.75f);
            batch.draw(capsule2,0,0);

            batch.setColor(0.25f,0.5f,0.25f,0.75f);
            batch.draw(capsule3,0,0);

            batch.setColor(0,0.25f,0.75f,0.75f);
            batch.draw(capsule4,0,0);

            batch.setColor(0.25f,0,0.5f,0.75f);
            batch.draw(capsule5,0,0);

            batch.setColor(0.3f,0.2f,0f,0.75f);
            batch.draw(capsule6,0,0);
        } else if (state == State.Triangulator){
            batch.setColor(0f, 0f, 1f, 1f);
            batch.draw(polyRegion, 300,250);
        } else if (state == State.Extruder){
            batch.setColor(0f, 0f, 1f, 0.5f);
            batch.draw(pathRegion1, 300,250);
            batch.draw(pathRegion2, 150,250);
            batch.draw(pathRegion3, 450,250);
        } else if(state == State.PathSmoother){
            drawText("Input points: "+inputNum, 25, 75);
            drawText("Output points: "+outputNum, 25, 50);
            if (smoothPoly != null){
                batch.setColor(0f, 0f, 1f, 0.5f);
                batch.draw(smoothPoly,0,0);
            }
        } else if (state == State.Spline){
            if (pathPoly != null){
                batch.setColor(1f, 0f, 0f, 0.5f);
                batch.draw(pathPoly, 0,0);
            }
            if(splinePoly != null){
                batch.setColor(0f, 0f, 1f, 0.5f);
                batch.draw(splinePoly, 0,0);
            }
        }

        batch.end();
    }

    /**
     * Called when the Application is destroyed.
     *
     * This is preceded by a call to pause().
     */
    @Override
    public void dispose () {
        batch.dispose();
    }

    /**
     * Draws text on the screen.
     *
     * @param text The string to draw
     * @param x The x-coordinate of the lower-left corner
     * @param y The y-coordinate of the lower-left corner
     */
    public void drawText(String text, float x, float y) {
        GlyphLayout layout = new GlyphLayout(displayFont,text);
        displayFont.setColor(Color.BLACK);
        displayFont.draw(batch, layout, x, y);
    }

    private void handleDrug(){
        float[] vert = getPoints();
        if (!active){
            smoother.set(vert);
            smoother.calculate();
            vert = smoother.getPoints();
            outputNum = vert.length/2;
        }
        extruder.set(vert, false);
        extruder.calculate(PATH_WIDTH);
        temp = extruder.getPolygon();

        smoothPoly = temp.makePolyRegion(region);
    }

    private void handleClick(){
        if (points.size()>2){
            float[] vert = getPoints();
            extruder.set(vert, false);
            extruder.calculate(LINE_WIDTH);
            temp = extruder.getPolygon();
            pathPoly = temp.makePolyRegion(region);

            if (points.size()/2 % 3 == 1){
                Spline2 spline2 = new Spline2(vert);
                pather = new SplinePather(spline2);
                pather.calculate();
                Path2 splinePath = pather.getPath();
                extruder.set(splinePath);
                extruder.calculate(LINE_WIDTH);
                temp = extruder.getPolygon();
                splinePoly = temp.makePolyRegion(region);
            }
        }
    }

    private float[] getPoints(){
        float[] vert = new float[points.size()];
        for (int i=0; i<points.size(); i++){
            vert[i] = points.get(i);
        }
        return vert;
    }

    @Override
    public boolean keyDown(int i) {
        if (i == Input.Keys.LEFT || i == Input.Keys.RIGHT){
            State[] states = State.values();
            int cur = state.ordinal();
            if (i == Input.Keys.LEFT){
                if (cur != 0){
                    cur --;
                } else{
                    cur = states.length-1;
                }
            } else{
                if (cur != states.length-1){
                    cur++;
                } else{
                    cur = 0;
                }
            }
            state = states[cur];
            initState(state);
        }
        return false;
    }

    @Override
    public boolean keyUp(int i) {
        return false;
    }

    @Override
    public boolean keyTyped(char c) {
        return false;
    }

    @Override
    public boolean touchDown(int i, int i1, int i2, int i3) {
        if (state == State.Spline || state == State.PathSmoother){
            if(state == State.PathSmoother){
                active = true;
                points = new ArrayList<>();
                inputNum = 1;
                outputNum = 0;
            }
            points.add((float) i);
            points.add((float) Gdx.graphics.getHeight()-i1);
        }
        return false;
    }

    @Override
    public boolean touchUp(int i, int i1, int i2, int i3) {
        if (active){
            active = false;
            handleDrug();
        }
        if (state == State.Spline){
            handleClick();
        }
        return false;
    }

    @Override
    public boolean touchDragged(int i, int i1, int i2) {
        if(active && state == State.PathSmoother){
            points.add((float) i);
            points.add((float) Gdx.graphics.getHeight()-i1);
            inputNum +=1;
            handleDrug();
        }
        return false;
    }

    @Override
    public boolean mouseMoved(int i, int i1) {
        return false;
    }

    @Override
    public boolean scrolled(float v, float v1) {
        return false;
    }
}
