/*
 * MusicBufferParser.java
 *
 * This is an interface for parsing a JSON entry into a MusicBuffer asset. It allows
 * you to create sample sequences ahead of time. 
 *
 * @author Walker M. White
 * @data   04/20/2020
 */
package edu.cornell.gdiac.assets;

import com.badlogic.gdx.assets.AssetManager;
import com.badlogic.gdx.utils.JsonValue;
import com.badlogic.gdx.utils.ObjectMap;
import edu.cornell.gdiac.audio.MusicQueue;

/**
 * This class parses a JSON entry into a {@link MusicQueue}.
 *
 * The music buffer created does not need to contain any {@link AudioSource} objects.
 * However, you can initialize sources in the sources tag with an array of file names. 
 */
public class MusicQueueParser implements AssetParser<MusicQueue> {
    /** The current font entry in the JSON directory */
    private JsonValue root;

    /**
     * Returns the asset type generated by this parser
     *
     * @return the asset type generated by this parser
     */
    public Class<MusicQueue> getType() {
        return MusicQueue.class;
    }

    /**
     * Resets the parser iterator for the given directory.
     *
     * The value directory is assumed to be the root of a larger JSON structure.
     * The individual assets are defined by subtrees in this structure.
     *
     * @param directory    The JSON representation of the asset directory
     */
    public void reset(JsonValue directory) {
        root = directory;
        root = root.getChild( "soundtrack" );
    }

    /**
     * Returns true if there are still assets left to generate
     *
     * @return true if there are still assets left to generate
     */
    public boolean hasNext() {
        return root != null;
    }

    /**
     * Processes the next available music buffer, loading it into the asset manager
     *
     * The parser converts JSON entries into {@link MusicQueueLoader.MusicBufferParameters}
     * values of the same name. The file will be the contents of the file entry.  The 
     * key will be the name of the font object.
     *
     * This method fails silently if there are no available assets to process.
     *
     * @param manager    The asset manager to load an asset
     * @param keymap    The mapping of JSON keys to asset file names
     */
    public void processNext(AssetManager manager, ObjectMap<String,String> keymap) {
        MusicQueueLoader.MusicQueueParameters params = new MusicQueueLoader.MusicQueueParameters();
        params.isMono = root.getBoolean( "mono",false );
        params.sampleRate = root.getInt( "sampleRate",44100 );
        params.volume = root.getFloat( "volume", 1.0f );
        params.pitch = root.getFloat( "pitch", 1.0f );
        params.panning = root.getFloat( "panning", 0.0f );
        params.looping = root.getBoolean( "looping", false );
        params.shortLoop = root.getBoolean( "shortLoop", false );
        JsonValue sources = root.get("sources");
        if (sources != null) {
            for (int ii = 0; ii < sources.size; ii++) {
                params.sources.add( sources.getString(ii));
            }
        }
        keymap.put(root.name(), root.name());
        manager.load( root.name(), MusicQueue.class, params );
        root = root.next();
    }

    /**
     * Returns true if o is another MusicBufferParser
     *
     * @return true if o is another MusicBufferParser
     */
    public boolean equals(Object o) {
        return o instanceof MusicQueueParser;
    }

}