/*
 * GDRoot.java
 * 
 * Simple application to show off how a memory pool works.
 * 
 * Because this is a simple demonstration, we have removed a lot of the overhead that 
 * we saw in the labs.  There is no canvas (we just use a direct SpriteBatch).  There 
 * are no subcontrollers.  There is no input controller.  There is no asset loader.  
 * We just leave everything bare and out in the open.  This should give you some
 * more appreciation of our class separation in the labs.
 *
 * Author: Walker M. White
 * Version: 2/24/2015
 */
package edu.cornell.gdiac.memory;

import com.badlogic.gdx.*;
import com.badlogic.gdx.utils.*;
import com.badlogic.gdx.graphics.*;
import com.badlogic.gdx.graphics.g2d.*;
import com.badlogic.gdx.math.MathUtils;


/**
 * Root class for a LibGDX.  
 * 
 * This class is technically not the ROOT CLASS. Each platform has another class above
 * this (e.g. PC games use DesktopLauncher) which serves as the true root.  However, 
 * those classes are unique to each platform, while this class is the same across all 
 * plaforms. In addition, this functions as the root class all intents and purposes, 
 * and you would draw it as a root class in an architecture specification.  
 */
public class GDXRoot extends ApplicationAdapter {
	/** Respawn rate of the particles */
	public static final int PARTICLE_RESPAWN = 4;
	
	/**  GameCanvas (VIEW).  This is much lower level than the labs. */
	private SpriteBatch canvas;

	/** Collection of particle objects (MODEL) */
	private ObjectSet<Particle> particles;

	/** Texture image for the photon (???) */
	private Texture texture;
	
	/** Memory pool for (pre)allocation of particles (???) */
	private ParticlePool memory;

	/** Simple field to slow down the allocation of photons */
	private int cooldown = 0;
	
	/**
	 * Creates a new game from the configuration settings.
	 *
	 * This method creates all of the objects, including preallocation of
	 * the particles.
	 */
	@Override
	public void create () {
		canvas  = new SpriteBatch();
		canvas.getProjectionMatrix().setToOrtho2D(0, 0, Gdx.graphics.getWidth(), Gdx.graphics.getHeight());
		texture = new Texture("photon.png");
		particles = new ObjectSet<Particle>();
		memory = new ParticlePool();
	}

	/**
	 * Renders a single animation frame.
	 * 
	 * This method executes the core loop.  We divide it up in to the two parts, to 
	 * keep them logically separate.
	 */
	@Override
	public void render () {
		update();
		draw();
	}
	
	/** 
	 * Called when the Application is destroyed. 
	 *
	 * This is preceded by a call to pause().
	 */
	@Override
	public void dispose() {
		memory.clear();
		texture.dispose();
	}
	
	/**
	 * Updates the status of all of the game objects.
	 * 
	 * This method generates particles according to user input, garbage collects them, and
	 * moves them.
	 */
	public void update() {
		// Garbage collect any objects that go offscreen
		for(Particle item : particles) {
			if (item.getX() < 0 || item.getX() > Gdx.graphics.getWidth() ||
				item.getY() < 0 || item.getY() > Gdx.graphics.getHeight()) {
				particles.remove(item);
				memory.free(item);
			}
		}
		
		
		// Generate new particles if the mouse is pressed.
		if (Gdx.input.isTouched(0) && cooldown <= 0) {
			// Add a particle to the set
			Particle item = memory.obtain();

			// Only proceed if allocation succeeded.
			if (item != null) {
				// Initialize the object
				float angle = MathUtils.random()*MathUtils.PI2;
				// MOUSE LOCATION HAS INVERTED Y-VALUE!
				item.getPosition().set(Gdx.input.getX(), Gdx.graphics.getHeight()-Gdx.input.getY());
				item.setAngle(angle);
				cooldown = PARTICLE_RESPAWN;
				// Add it to the set of objects
				particles.add(item);
			}
		}
		
		// Move all particles
		for(Particle item : particles) {
			item.move();
		}	
		
		// Reset cooldown if necessary
		if (cooldown > 0) {
			cooldown--;
		}
	}

	/**
	 * Draw all of the game objects.
	 * 
	 * This method calls raw SpriteBatch commands.
	 */
	public void draw() {
		// Clear the screen at the start of the animation frame
		Gdx.gl.glClearColor(0.39f, 0.58f, 0.93f, 1.0f);  // Homage to the XNA years
		Gdx.gl.glClear(GL20.GL_COLOR_BUFFER_BIT);
		
		// Animate all of the photons.  
		canvas.begin();
		canvas.setBlendFunction(GL20.GL_SRC_ALPHA, GL20.GL_ONE); // Additive blending
		for(Particle item : particles) {
			// Draw the object centered at x.
			canvas.draw(texture, item.getX()-texture.getWidth()/2, item.getY()-texture.getHeight()/2);
		}
		canvas.end();
	}
}
