structure LambdaOpt = struct
  open Lambda

  (* increment variables numbered >= i by j *)
  fun inc(i:int,j:int,e:lexp):lexp =
    case e of
	  Int_l _ => e
      | Real_l _ => e
      | Var_l k => if k >= i then Var_l(k+j) else e
      | Fn_l(e) => Fn_l(inc(i+1,j,e))
      | App_l(e3,e4) => App_l(inc(i,j,e3),inc(i,j,e4))
      | Unop_l(u,e) => Unop_l(u,inc(i,j,e))
      | Binop_l(e3,b,e4) => Binop_l(inc(i,j,e3),b,inc(i,j,e4))
      | Tuple_l(es) => Tuple_l(map (fn e => inc(i,j,e)) es)
      | Ith_l(e3,e4) => Ith_l(inc(i,j,e3), inc(i,j,e4))
      | SetIth_l(e3,e4,e5) =>
	  SetIth_l(inc(i,j,e3), inc(i,j,e4), inc(i,j,e5))
      | If_l(e3,e4,e5) =>
	  If_l(inc(i,j,e3), inc(i,j,e4), inc(i,j,e5))
      | Letrec_l(e3,e4) => Letrec_l(inc(i+2,j,e3),inc(i+1,j,e4))
      | Error_l _ => e


  (* substitute e1 for variable i within e2 *)
  fun subst(e1:lexp, i:int, e2:lexp):lexp =
    case e2 of
	  Int_l _ => e2
      | Real_l _ => e2
      | Var_l j =>
	    if (i = j) then e1 else if (i < j) then Var_l(j-1) else Var_l j
      | Fn_l(e) => Fn_l(subst(inc(0,1,e1),i+1,e))
      | App_l(e3,e4) => App_l(subst(e1,i,e3), subst(e1,i,e4))
      | Unop_l(u,e) => Unop_l(u,subst(e1,i,e))
      | Binop_l(e3,b,e4) => Binop_l(subst(e1,i,e3),b,subst(e1,i,e4))
      | Tuple_l(es) => Tuple_l(map (fn e => subst(e1,i,e)) es)
      | Ith_l(e3,e4) => Ith_l(subst(e1,i,e3), subst(e1,i,e4))
      | SetIth_l(e3,e4,e5) =>
	  SetIth_l(subst(e1,i,e3), subst(e1,i,e4), subst(e1,i,e5))
      | If_l(e3,e4,e5) =>
	  If_l(subst(e1,i,e3), subst(e1,i,e4), subst(e1,i,e5))
      | Letrec_l(e3,e4) =>
	  let val e1_inc = inc(0,1,e1)
	  in
	      Letrec_l(subst(inc(0,1,e1_inc),i+2,e3),subst(e1_inc,i+1,e4))
	  end
      | Error_l _ => e2

  (* does subexpression have no side effects? *)
  fun ok_to_inline(e:lexp):bool =
    case e of
	  Int_l _ => true
      | Real_l _ => true
      | Var_l _ => true
      | Fn_l e1 => ok_to_inline e1
      | App_l(e1,e2) => ok_to_inline e1 andalso ok_to_inline e2
      | Unop_l(_,e1) => ok_to_inline e1
      | Binop_l(e1,_,e2) => ok_to_inline e1 andalso ok_to_inline e2
      | Tuple_l es => List.all ok_to_inline es
      | Ith_l(e1,e2) => ok_to_inline e1 andalso ok_to_inline e2
      | SetIth_l _ => false (* side effect *)
      | If_l(e1,e2,e3) => ok_to_inline(e1) andalso
	    ok_to_inline(e2) andalso ok_to_inline(e3)
      | Letrec_l _ => false
      | Error_l _ => false

  (* number of occurrences of a var in an exp *)
  fun num_occurs(i:int, e:lexp):int =
    case e of
	  Int_l _ => 0
      | Real_l _ => 0
      | Var_l j => if (i = j) then 1 else 0
      | Fn_l(e) => num_occurs(i+1,e)
      | App_l(e3,e4) => num_occurs(i,e3) + num_occurs(i,e4)
      | Unop_l(u,e) => num_occurs(i,e)
      | Binop_l(e3,b,e4) => num_occurs(i,e3) + num_occurs(i,e4)
      | Tuple_l(es) => foldl (fn (e,s) => s + num_occurs(i,e)) 0 es
      | Ith_l(e3,e4) => num_occurs(i,e3) + num_occurs(i,e4)
      | SetIth_l(e3,e4,e5) =>
	    num_occurs(i,e3) + num_occurs(i,e4) + num_occurs(i,e5)
      | If_l(e3,e4,e5) =>
	    num_occurs(i,e3) + num_occurs(i,e4) + num_occurs(i,e5)
      | Letrec_l(e3,e4) => num_occurs(i+2,e3) + num_occurs(i+1,e4)
      | Error_l _ => 0

  val simplified = ref true
  fun simplify(e2:lexp):lexp =
    case e2 of
  	Int_l _ => e2
      | Real_l _ => e2
      | Var_l j => e2
      | Fn_l(e) => Fn_l(simplify(e))
      | App_l(Fn_l(e1),e2) =>
	    if ok_to_inline(e2) andalso num_occurs(0,e1) <= 1 then
	     (simplified := true; print (".");
	      simplify(subst(e2,0,e1)))
	    else App_l(Fn_l(simplify(e1)),simplify(e2))
      | App_l(e3,e4) => App_l(simplify(e3),simplify(e4))
      | Unop_l(u,e) => Unop_l(u,simplify(e))
      | Binop_l(e3,b,e4) => Binop_l(simplify(e3),b,simplify(e4))
      | Tuple_l(es) => Tuple_l(map simplify es)
      | Ith_l(e3,e4) => Ith_l(simplify(e3), simplify(e4))
      | SetIth_l(e3,e4,e5) =>
	    SetIth_l(simplify(e3), simplify(e4), simplify(e5))
      | If_l(e3,e4,e5) =>
	    If_l(simplify(e3), simplify(e4), simplify(e5))
      | Letrec_l(e3,e4) => Letrec_l(simplify(e3), simplify(e4))
      | Error_l _ => e2

  fun optimize(e:lexp):lexp =
      let fun loop(e) =
	  if (!simplified) then (simplified := false; loop(simplify(e)))
	  else e
      in
	  print ("Optimizing...");
	  simplified := false;
	  let val e = loop(simplify(e))
	  in
	    print("\n");
	    e
	  end
      end

end
