signature LAMBDA_CSE = sig

    (* node depth * subtree has value hash * prefix *)
    type lexp_cse_data = (int * int * int list)

    datatype lexp_cse_node =
        Int_c of int
      | Real_c of real
      | Var_c of int
      | Fn_c of lexp_cse
      | App_c of lexp_cse * lexp_cse
      | Unop_c of Lambda.lunop * lexp_cse
      | Binop_c of lexp_cse  * Lambda.lbinop * lexp_cse
      | Tuple_c of lexp_cse list
      | Ith_c of lexp_cse  * lexp_cse
      | SetIth_c of lexp_cse  * lexp_cse  * lexp_cse
      | If_c of lexp_cse  * lexp_cse  * lexp_cse
      | Letrec_c of lexp_cse  * lexp_cse
      | Error_c of string
    and lexp_cse = CSENode of lexp_cse_node * lexp_cse_data

    (* converts the entire lexp_cse to a tree
       booleans are for printing depth, hash, and prefix *)
    val cseToString: lexp_cse * bool * bool * bool -> string
    (* converts a Lambda tree to a CSE tree, adding the
       depth and prefix information (hash is added seperately,
       default hash is ~1) *)
    val lexpToCSE: Lambda.lexp -> lexp_cse
    (* converts a CSE tree to a Lambda tree, throwing away the data *)
    val cseTolexp: lexp_cse -> Lambda.lexp

end

structure LambdaCSE :> LAMBDA_CSE = struct

    open Lambda

     (* depth * hash * prefix *)
     type lexp_cse_data = (int * int * int list)

     datatype lexp_cse_node = 
	    Int_c of int 
      | Real_c of real 
      | Var_c of int 
      | Fn_c of lexp_cse 
      | App_c of lexp_cse * lexp_cse 
      | Unop_c of lunop * lexp_cse  
      | Binop_c of lexp_cse  * lbinop * lexp_cse 
      | Tuple_c of lexp_cse list 
      | Ith_c of lexp_cse  * lexp_cse  
      | SetIth_c of lexp_cse  * lexp_cse  * lexp_cse 
      | If_c of lexp_cse  * lexp_cse  * lexp_cse 
      | Letrec_c of lexp_cse  * lexp_cse 
      | Error_c of string 
     and lexp_cse = CSENode of lexp_cse_node * lexp_cse_data 

    fun unop2s (u: lunop):string =
      case u of
        Neg_l => "~"
      | Not_l => "not"
      | Ptr_l => "ptr?"

    fun binop2s (b: lbinop):string =
        case b of
	    Plus_l => "+"
          | Minus_l => "-"
          | Times_l => "*"
          | Concat_l => "^"
          | Equal_l => "="
          | GreaterThan_l => ">"
          | LessThan_l => "<"

    fun mapAllButLast (f:'a -> 'a) (s:'a list) : 'a list =
      case s of
        [] => []
      | [x] => [x]
      | x::t => (f x)::(mapAllButLast f t)

    fun pixToString(pix:int list) = let
      val s = map Int.toString pix
      val t = mapAllButLast (fn x => x ^ ",") s
    in
      "[" ^ concat t ^ "]"
    end

    fun cseToString(c:lexp_cse,
                    withDepth:bool,
                    withHash:bool,
                    withPix:bool):string =
      let
        fun cseToString'(c:lexp_cse):string = let
          val CSENode(node,(depth,hash,pix)) = c
          val spaces = "                                                       "
          val prefix = String.extract(spaces,0,SOME(length pix))
          val depthString =
            if withDepth then "depth=" ^ Int.toString depth ^ " " else ""
          val hashString =
            if withHash then "hash=" ^ Int.toString hash ^ " " else ""
          val pixString =
            if withPix then "pix=" ^ pixToString pix ^ " " else ""
          val prefix = prefix ^ depthString ^ hashString ^ pixString
          val s =
            case node of
              Int_c(i) => Int.toString i ^ "\n"
            | Real_c(r) => Real.toString r ^ "\n"
            | Var_c(i) => "[" ^ Int.toString i ^ "]\n"
            | Fn_c(b) => "lambda\n" ^ cseToString' b
            | App_c(f,e) => "app\n" ^ cseToString' f ^ cseToString' e
            | Unop_c(u,e) => unop2s u ^ "\n" ^ cseToString' e
            | Binop_c(e1,b,e2) => binop2s b ^ "\n"
                                  ^ cseToString' e1
                                  ^ cseToString' e2
            | Tuple_c(es) => "Tuple\n" ^ concat(map (fn x => cseToString' x) es)
            | Ith_c(e1, e2) => "Ith\n" ^ cseToString' e1 ^ cseToString' e2
            | SetIth_c(e1,e2,e3) => "SetIth\n"
                ^ cseToString' e1
                ^ cseToString' e2
                ^ cseToString' e3
            | If_c(e1,e2,e3) => "If\n"
                ^ cseToString' e1
                ^ cseToString' e2
                ^ cseToString' e3
            | Letrec_c(e1,e2) => "Letrec\n" ^ cseToString' e1 ^ cseToString' e2
            | Error_c(s) => "Error " ^ s ^ "\n"
        in
          prefix ^ s
        end
      in
        cseToString' c
      end


     fun cseTolexp (CSENode(cNode,d): lexp_cse): lexp = 
       case cNode of 
           Int_c(i)    => Int_l(i)
        |  Real_c(r)   => Real_l(r)
        |  Var_c(i)    => Var_l(i)
        |  Fn_c(b)     => Fn_l(cseTolexp(b))
        |  App_c(f,e)           => App_l(cseTolexp(f), cseTolexp(e))
        |  Unop_c(u,e)          => Unop_l(u, cseTolexp(e))
        |  Binop_c(e1, b, e2)   => Binop_l(cseTolexp(e1), b, 
                                           cseTolexp(e2))
        |  Tuple_c(es)          => Tuple_l(map (fn(x)=> cseTolexp(x)) es)
        |  Ith_c(e1, e2)        => Ith_l(cseTolexp(e1), cseTolexp(e2))
        |  SetIth_c(e1, e2, e3) => SetIth_l(cseTolexp(e1),
                                            cseTolexp(e2),
                                            cseTolexp(e3))
        |  If_c(e1, e2, e3)     => If_l(cseTolexp(e1), 
                                        cseTolexp(e2), 
                                        cseTolexp(e3))
        |  Letrec_c(e1, e2)     => Letrec_l(cseTolexp(e1), 
                                            cseTolexp(e2))
        |  Error_c(s)           => Error_l(s)

     fun lexpToCSE (expr: lexp): lexp_cse =
       let
         fun toCSErec (e: lexp, depth: int, 
                       pix: int list): lexp_cse =
           case e of
               Int_l(i)      => CSENode(Int_c(i), (depth, ~1, pix))
            |  Real_l(r)     => CSENode(Real_c(r), (depth, ~1, pix))
            |  Var_l(i)      => CSENode(Var_c (i), (depth, ~1, pix))
            |  Fn_l(b)       => let 
                                  val body = toCSErec(b, depth + 1,
                                                      pix@[1])
                                in
                                  CSENode(Fn_c(body), 
                                          (depth, ~1, pix))
                                end
            |  App_l(f,a)    => let
                                  val func = toCSErec(f, depth, 
                                                      pix@[1])
                                  val arg  = toCSErec(a, depth,
                                                      pix@[2])
                                in
                                  CSENode(App_c(func, arg),
                                                (depth, ~1, pix))
                                end
            |  Unop_l(u,e1)  => let
                                  val arg = toCSErec(e1, depth,
                                                     pix@[1])
                                in
                                  CSENode(Unop_c(u, arg),
                                          (depth, ~1, pix)) 
                                end
            |  Binop_l(e1, b, e2)
                             => let
                                  val e1 = toCSErec(e1,depth,
                                                    pix@[1])
                                  val e2 = toCSErec(e2,depth,
                                                    pix@[2])
                                in
                                  CSENode(Binop_c(e1, b, e2),
                                          (depth, ~1, pix))
                                end
            |  Tuple_l(l)    => let
                                  fun recDown(i:int, 
                                          eList:lexp list) =
                                    case eList of 
                                        nil => nil
                                     |  x::xs => 
                                         toCSErec(x, depth, 
                                                pix@[i])::recDown(i+1,xs)
                                  val newList = recDown (1,l)
                                in
                                  CSENode(Tuple_c(newList),
                                          (depth, ~1, pix))
                                end
            |  Ith_l(i,e1)   => let
                                  val i  = toCSErec(i, depth, 
                                                    pix@[1])
                                  val e1 = toCSErec(e1, depth, 
                                                    pix@[2])
                                in
                                  CSENode(Ith_c(i, e1),
                                          (depth, ~1, pix))
                                end
            |  SetIth_l(e1, e2, e3)   
                             => let
                                  val e1 = toCSErec(e1, depth,
                                                    pix@[1])
                                  val e2 = toCSErec(e2, depth,
                                                    pix@[2])
                                  val e3 = toCSErec(e3, depth,
                                                    pix@[3])
                                in
                                  CSENode(SetIth_c(e1, e2, e3),
                                          (depth, ~1, pix))
                                end
            |  If_l(e1, e2, e3)
                             => let
                                  val e1 = toCSErec(e1, depth,
                                                    pix@[1])
                                  val e2 = toCSErec(e2, depth,
                                                    pix@[2])
                                  val e3 = toCSErec(e3, depth,
                                                    pix@[3])
                                in
                                  CSENode(If_c(e1, e2, e3),
                                          (depth, ~1, pix))
                                end
            |  Letrec_l(e1, e2)
                             => let
                                  val e1 = toCSErec(e1, depth + 1,
                                                    pix@[1])
                                  val e2 = toCSErec(e2, depth,
                                                    pix@[2])
                                in
                                  CSENode(Letrec_c(e1, e2),
                                          (depth, ~1, pix))
                                end
            |  Error_l(err)  => CSENode(Error_c(err), 
                                        (depth, ~1, pix))
       in
         toCSErec(expr, 0, [])
       end 

  end


