structure Hash :> sig

  (* annotate a CSE tree with hash values *)
  val hash: LambdaCSE.lexp_cse -> LambdaCSE.lexp_cse

end = struct

  open Lambda
  open LambdaCSE

  val sqrtMaxInt = 32768

  (* without the mods, this would be a 1-1 and onto pairing function *)
  fun encodePair(x:int,y:int) : int =
    let val z = (x mod sqrtMaxInt + y mod sqrtMaxInt) mod sqrtMaxInt
    in z*(z+1) div 2 + x
    end

  val encodeList : int list -> int = foldr encodePair 0

  fun unopCode(u:lunop) : int =
    case u of
      Neg_l => 1
    | Not_l => 2
    | Ptr_l => 3

  fun binopCode(b:lbinop) : int =
    case b of
      Plus_l => 4
    | Times_l => 5
    | Minus_l => 6
    | Equal_l => 7
    | Concat_l => 8
    | GreaterThan_l => 9
    | LessThan_l => 10

  fun code(e:lexp_cse_node) : int =
    case e of
      Int_c _ => 11
    | Real_c _ => 12
    | Var_c _ => 13
    | Fn_c _ => 14
    | App_c _ => 15
    | Unop_c(u,_) => unopCode u
    | Binop_c(_,b,_) => binopCode b
    | Tuple_c _ => 16
    | Ith_c _ => 17
    | SetIth_c _ => 18
    | If_c _ => 19
    | Letrec_c _ => 20
    | Error_c _  => 21

  fun hash(CSENode(e,(depth,_,prefix)):lexp_cse) : lexp_cse =
    case e of
      Int_c n => CSENode(e,(depth,encodePair(code e,n),prefix))
    | Real_c r => let
        val frac = Real.trunc(Real.realMod r * 1000000000.0)
        val whole = Real.trunc r
        val hashReal = encodePair(whole,frac)
      in
        CSENode(e,(depth,encodePair(code e,hashReal),prefix))
      end
    | Var_c n => CSENode(e,(depth,encodePair(code e,0),prefix))
    | Fn_c e' => let
        val subNode as CSENode(_,(_,subHash,_)) = hash(e')
      in
        CSENode(Fn_c subNode,(depth,encodePair(code e,subHash),prefix))
      end
    | App_c (e1,e2) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
      in
        CSENode(App_c(subNode1,subNode2),
               (depth,encodeList[code e,subHash1,subHash2],prefix))
      end
    | Unop_c (u,e') => let
        val subNode as CSENode(_,(_,subHash,_)) = hash e'
      in
        CSENode(Unop_c(u,subNode),
               (depth,encodePair(code e,subHash), prefix))
      end
    | Binop_c (e1,b,e2) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
      in
        CSENode(Binop_c(subNode1,b,subNode2),
               (depth,encodeList[code e,subHash1,subHash2],prefix))
      end
    | Tuple_c ls => let
        val newNodes = map hash ls
        val newHashList = map (fn CSENode(_,(_,h,_)) => h) newNodes
      in
        CSENode(Tuple_c newNodes,
               (depth,encodeList(code e::newHashList),prefix))
      end
    | Ith_c (e1,e2) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
      in
        CSENode(Ith_c(subNode1,subNode2),
               (depth,encodeList [code e,subHash1,subHash2],prefix))
      end
    | SetIth_c (e1,e2,e3) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
        val subNode3 as CSENode(_,(_,subHash3,_)) = hash e3
        val newNodes = (subNode1,subNode2,subNode3)
        val newHashList = [code e,subHash1,subHash2,subHash3]
      in
        CSENode(SetIth_c newNodes,
               (depth,encodeList newHashList,prefix))
      end
    | If_c (e1,e2,e3) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
        val subNode3 as CSENode(_,(_,subHash3,_)) = hash e3
        val newNodes = (subNode1,subNode2,subNode3)
        val newHashList = [code e,subHash1,subHash2,subHash3]
      in
        CSENode(If_c newNodes,(depth,encodeList newHashList,prefix))
      end
    | Letrec_c (e1,e2) => let
        val subNode1 as CSENode(_,(_,subHash1,_)) = hash e1
        val subNode2 as CSENode(_,(_,subHash2,_)) = hash e2
      in
        CSENode(Letrec_c(subNode1,subNode2),
               (depth,encodeList[code e,subHash1,subHash2],prefix))
      end
    | Error_c s => let
        val hashString = encodeList (map Char.ord (explode s))
      in
        CSENode(e,(depth,encodePair(code e,hashString), prefix))
      end

end