signature MEMORY = sig
  val max_memory_size : int
  val g1_start : int
  val g2a_start : int
  val g2b_start : int
  val memory : Value.value array
  val startptr : int ref
  val allocptr : int ref
  val limitptr : int ref
  val g2_startptr : int ref
  val g2_allocptr : int ref
  val g2_limitptr : int ref
  val dynamic_env : Value.value ref
  val print_memory : unit -> unit
  val reset : unit -> unit
end

structure Memory :> MEMORY = struct
  (* The memory for the abstract machine *)
  val max_memory_size : int = 1500;
  val g1_start : int = 0;
  val g2a_start : int = max_memory_size div 3;
  val g2b_start : int = g2a_start * 2;
  
  val memory : Value.value array =
      Array.array (max_memory_size, Value.Int_v(0));



  (* start of current allocation space *)
  val startptr : int ref = ref g1_start;
  (* next available word *)
  val allocptr : int ref = ref g1_start;
  (* first unavailable word *)
  val limitptr : int ref = ref g2a_start;


  (* start of g2 allocation space *)
  val g2_startptr : int ref = ref g2a_start;
  (* next available word *)
  val g2_allocptr : int ref = ref g2a_start;
  (* first unavailable word *)
  val g2_limitptr : int ref = ref g2b_start;


  
  (* start of a linked list containing the
   * values for all of the variables currently in scope *)
  val dynamic_env : Value.value ref = ref (Value.Int_v 0)

  (* reset memory *)
  fun reset() : unit =
    (startptr := g1_start;
     allocptr := g1_start;
     limitptr := g2a_start;
	 g2_startptr := g2a_start;
	 g2_allocptr := g2a_start;
	 g2_limitptr := g2b_start;

     Array.modify (fn _ => Value.Int_v 0) memory;
     dynamic_env := (Value.Int_v 0))

  

  
  (* print out a range of memory from start to stop *)
  fun print_memory_range(start:int, stop:int):unit =
      let fun loop(i:int) =
	  if i < (stop) then
	      (print (Int.toString i); print ":";
	       Value.print_value(Array.sub(memory,i));
	       print "\n";
	       loop (i+1))
	  else ()
      in
	  print "memory dump: start="; print (Int.toString(start));
	  print "\n------------------------------------\n";
	  loop(start);
	  print "-------------------------------------\n"
      end
  
  fun print_memory():unit = 
	(print_memory_range(!startptr,!allocptr);
	 print_memory_range(!g2_startptr,!g2_allocptr))
	
end
