structure Lambda :> LAMBDA =
  struct
    (* datatypes that define a core, lambda-calculus-based language *)

    (* unary operations:  same as in AbSyn except that Ptr_l returns
     * "true" when the object being tested is a pointer, and otherwise
     * returns "false". *)
    datatype lunop = Neg_l | Not_l | Ptr_l

    (* binary operations:  same as in AbSyn *)
    datatype lbinop =
	Plus_l | Times_l | Minus_l | Equal_l | Concat_l | GreaterThan_l |
	LessThan_l

    (* lambda expressions:  Most of the expression forms are similar
     * to those in AbSyn.  However, Lambda doesn't have identifiers.
     * Instead, variables are represented as integers.  The convention
     * is that Var_l(0) corresponds to the nearest enclosing bound variable,
     * Var_l(1) corresponds to the next outer enclosing bound variable,
     * Var_l(2) to the next, etc.  So, for example we represent the
     * function "(fn x => (fn y => x + y))" as follows:
     *
     *     Fn_l(Fn_l(Binop_l(Var_l(1),Plus_l,Var_l(0))))
     *
     * Notice that in the sub-expression "x + y", y is the nearest
     * enclosing bound variable and so it is represented as Var_l(0),
     * and x is the next nearest enclosing bound variable so it is
     * represented as Var_l(1).
     *
     * The other big change from MiniML is that there is no pattern
     * matching.  Instead, there is only a primitive if-then-else
     * construct (If_l).  In addition, there are no chars, no strings,
     * no records, no datatypes, etc.  Instead, we must compile these
     * features out. So, for instance, we use tuples to represent
     * records, and a combination of tuples and integers to represent
     * datatypes, integers to represent chars, and tuples of integers
     * to represent strings.
     *)
    datatype lexp =
	Int_l of int
      | Real_l of real
      | Var_l of int
      | Fn_l of lexp
      | App_l of lexp * lexp
      | Unop_l of lunop * lexp
      | Binop_l of lexp * lbinop * lexp
      | Tuple_l of lexp list
      | Ith_l of lexp * lexp
      | SetIth_l of lexp * lexp * lexp
      | If_l of lexp * lexp * lexp
      | Letrec_l of lexp * lexp
      | Error_l of string

    val True_l = Int_l(1)
    val False_l = Int_l(0)
    fun And_l(l1,l2) = If_l(l1,l2,False_l)
    fun Or_l(l1,l2) = If_l(l1,True_l,l2)

    structure P = Compiler.PrettyPrint

    fun sep (pp:P.ppstream) (s:unit->unit) (l:(unit->unit) list):unit =
      case l of
        [] => ()
      | [x] => x ()
      | (x::xs) => (x (); s (); sep pp s xs)

    val max_prec = 999;

    fun binop_prec(b:lbinop):int =
        case b of
    	    Plus_l => 7
          | Times_l => 8
          | Minus_l => 7
          | Concat_l => 7
          | Equal_l => 5
          | GreaterThan_l => 5
          | LessThan_l => 5

    fun prec(e:lexp):int =
        case e of
	    Int_l(_) => max_prec
	  | Real_l(_) => max_prec
          | Var_l(_) => max_prec
          | Fn_l(e) => 1
          | App_l(e1,e2) => 2
          | Unop_l (u,e) => 2
          | Binop_l(e1,b,e2) => binop_prec(b)
          | Tuple_l(es) => max_prec
          | Ith_l(e1,e2) => 2
	  | SetIth_l(e1,e2,e3) => 2
	  | If_l(e1,e2,e3) => 1
	  | Letrec_l(e1,e2) => max_prec
          | Error_l(_) => max_prec

    fun unop2s (u: lunop):string =
      case u of
        Neg_l => "~"
      | Not_l => "not"
      | Ptr_l => "ptr?"

    fun binop2s (b: lbinop):string =
        case b of
	    Plus_l => "+"
          | Minus_l => "-"
          | Times_l => "*"
          | Concat_l => "^"
          | Equal_l => "="
          | GreaterThan_l => ">"
          | LessThan_l => "<"

    val empty_senv : string list = []
    fun next_var chars =
	case chars of
	    [] => ""
	  | hd::tl =>
		if (hd < #"z") then implode(chr(ord(hd)+1)::tl)
		else implode(#"a"::chars)

    fun push_var (senv: string list):string list =
	case senv of
	    [] => ["a"]
	  | hd::tl => (next_var (explode hd))::senv

    fun get_var (senv:string list) (i:int) =
	List.nth(senv,i) handle _ => ("env["^(Int.toString(i))^"]")

    fun exp2s (pp:P.ppstream) (p:int) (senv:string list) (e:lexp):unit = let
      fun begin ():unit = P.begin_block pp P.INCONSISTENT 0
      fun stop ():unit = P.end_block pp
      fun brI ():unit = P.add_break pp (1,2)
      fun br ():unit = P.add_break pp (1,0)
      fun str (s:string):unit = P.add_string pp s
      val p' = prec(e)
      val e2s = exp2s pp p'
      fun s pp =
        case e of
	  Int_l(i) => str (Int.toString i)
	| Real_l(r) => str (Real.toString r)
        | Var_l(x) => str (get_var senv x)
        | Fn_l(e) =>
	      let val senv' = push_var senv
	      in
		  (begin ();
		   str "(fn ";
		   str (get_var senv' 0);
		   str " =>";
	           brI ();
		   e2s senv' e;
		   str ")";
		   stop ())
	      end
        | App_l(e1,e2) => (begin ();
			   e2s senv e1;
			   brI ();
			   e2s senv e2;
			   stop ())
        | Unop_l (u,e) => (begin ();
			   str (unop2s u);
			   brI ();
			   e2s senv e;
			   stop ())
        | Binop_l(e1,b,e2) => (begin ();
			       e2s senv e1;
			       str " ";
			       str (binop2s b);
			       brI ();
			       e2s senv e2;
			       stop ())
        | Tuple_l(es) => (str "(";
                            begin ();
                            sep pp (fn () => (str ","; br ()))
                                   (List.map (fn (e) =>
                                                fn () => exp2s pp 0 senv e)
                                             es);
                            stop ();
                            str ")")
        | Ith_l(e1,e2) => (begin ();
			   str "#";
			   e2s senv e1;
			   str " ";
			   e2s senv e2;
			   stop ())
	| SetIth_l(e1,e2,e3) => (begin();
				 e2s senv e2;
				 str "[";
				 begin();
				 e2s senv e1;
				 stop();
				 str "] := ";
				 e2s senv e3;
				 stop())
        | If_l(e1,e2,e3) => (begin ();
			     str "if ";
			     e2s senv e1;
			     str " then "; brI();
			     e2s senv e2;  br();
			     str " else "; brI();
			     e2s senv e3;
			     stop ())
        | Letrec_l(e1,e2) =>
	    let val senv1 = push_var senv
		val senv2 = push_var senv1
		val f = get_var senv2 1
		val x = get_var senv2 0
	    in
            (begin ();
             str "letrec ";
	     str f;
	     str "(";
	     str x;
	     str ") = "; brI();
	     exp2s pp 0 senv2 e1; br();
             str " in "; brI();
             exp2s pp 0 senv1 e2; br();
             str " end ";
             stop ())
	    end
	| Error_l(s) => str ("error("^s^")")
    in
      if (p' > p) then s pp  else (str "("; s pp; str ")")
    end

    fun ppLexp (e:lexp):string = let
      fun ppExp' (p:P.ppstream) (e:lexp):unit = exp2s p 0 [] e;
    in
      P.pp_to_string 80 ppExp' e
    end

  end