signature LAMBDA = 
  sig
    (* datatypes that define a core, lambda-calculus-based language *)

    (* unary operations:  same as in AbSyn except that Ptr_l returns
     * "true" when the object being tested is a pointer, and otherwise
     * returns "false". *)
    datatype lunop = Neg_l | Not_l | Ptr_l

    (* binary operations:  same as in AbSyn *)
    datatype lbinop = 
	Plus_l | Times_l | Minus_l | Equal_l | Concat_l | GreaterThan_l |
	LessThan_l

    (* lambda expressions:  Most of the expression forms are similar
     * to those in AbSyn.  However, Lambda doesn't have identifiers.
     * Instead, variables are represented as integers.  The convention
     * is that Var_l(0) corresponds to the nearest enclosing bound variable,
     * Var_l(1) corresponds to the next outer enclosing bound variable,
     * Var_l(2) to the next, etc.  So, for example we represent the
     * function "(fn x => (fn y => x + y))" as follows:
     * 
     *     Fn_l(Fn_l(Binop_l(Var_l(1),Plus_l,Var_l(0))))
     *
     * Notice that in the sub-expression "x + y", y is the nearest
     * enclosing bound variable and so it is represented as Var_l(0),
     * and x is the next nearest enclosing bound variable so it is
     * represented as Var_l(1). 
     *
     * The other big change from MiniML is that there is no pattern
     * matching.  Instead, there is only a primitive if-then-else
     * construct (If_l).  In addition, there are no chars, no strings, 
     * no records, no datatypes, etc.  Instead, we must compile these
     * features out. So, for instance, we use tuples to represent 
     * records, and a combination of tuples and integers to represent
     * datatypes, integers to represent chars, and tuples of integers
     * to represent strings.
     *)
    datatype lexp = 
	Int_l of int
      | Real_l of real
      | Var_l of int
      | Fn_l of lexp
      | App_l of lexp * lexp
      | Unop_l of lunop * lexp
      | Binop_l of lexp * lbinop * lexp
      | Tuple_l of lexp list
      | Ith_l of lexp * lexp
      | SetIth_l of lexp * lexp * lexp
      | If_l of lexp * lexp * lexp
      | Letrec_l of lexp * lexp
      | Error_l of string

    (* a few useful abbreviations *)
    val True_l:lexp 
    val False_l:lexp
    val And_l: lexp*lexp -> lexp
    val Or_l: lexp*lexp -> lexp

    val ppLexp : lexp -> string
  end


