structure LambdaInterp = struct
  structure L = Lambda
  open Value
  structure M = Memory
  structure S = Stack

  val push = S.push
  val pop = S.pop
  val dynamic_env = M.dynamic_env

  (* set to true to automatically print the stack and the heap *)
  val debug = ref false

  (* raised when an error occurs during interpretation *)
  val error = Error.runtime

  (* allocate space for an object of size s, returning a pointer
   * into the memory.  Does a gc if needed to free up space. *)
  fun malloc (s:int):int =
      let val a = !M.allocptr
      in
	  if a + s < !M.limitptr then
	      (M.allocptr := a + s; a)
	  else (push (!dynamic_env);
		GC.gc s; (* need s words *)
		dynamic_env := pop();
		malloc s)
      end

  (* initialize memory at offset p with the values in the list vs.  Also
   * puts in the tag word. *)
  fun init_object (p:int) (vs:value list) : unit =
      let fun init (p:int) (vs:value list) : unit =
	  case vs of
	      [] => ()
	    | v::rest => (Array.update(M.memory,p,v); init (p+1) rest)
      in
	  Array.update(M.memory,p,Tag_v(length vs));
	  init (p+1) vs
      end

  (* allocate and initialize a tuple of values *)
  fun malloc_tuple(vs:value list):value =
      (* save all of the values in case gc occurs *)
      (app push vs;
       let val p = malloc(length vs + 1) (* one extra for tag *)
	   (* restore values from stack *)
	   val vs = rev(map (fn _ => pop()) vs)
       in
	   init_object p vs;
	   Ptr_v p
       end)

  (* get the ith word in a tuple in the memory *)
  fun get_ith(v:value,i:int):value =
      case (v) of
	  Ptr_v(p) => Array.sub(M.memory,p+i)
	| _ => error("get_ith")

  (* set the ith word in a tuple in the memory to v' *)
  fun set_ith(v:value,i:int,v':value):unit =
      case (v) of
	  Ptr_v(p) => Array.update(M.memory,p+i,v')
	| _ => error("set_ith")

  (* given a linked list in the memory starting at v, get the offset'th
   * element in the list.
   *)
  fun lookup(v:value,offset:int) =
      case offset of
	  0 => get_ith(v,1)
	| _ => lookup(get_ith(v,2),offset-1)

  (* the following 3 functions help the string binop operations *)
  fun readString(p:int):value list =
    case get_ith(Ptr_v(p),0) of
      Tag_v(l1) => List.tabulate(l1,(fn(n) => (get_ith(Ptr_v(p),n+1))))
      | _ => error("string tag expected")
  fun valToChar(Int_v(i)) = Char.chr i
    | valToChar(_) = error("bad val")
  (* [Int(i)...]->[int...]->[char...]->string *)
  fun convertString(vl:value list) =
    String.implode ((tl( tl (map valToChar vl))) 
                    handle e => error("bad string"))
  fun printString(p:int):string = convertString(readString p)

  (* evaluate the expression e in the context of the current
   * dynamic environment.
   *)
  fun eval (e:L.lexp) : value =
    case e of
      L.Int_l(i) => Int_v(i)
    | L.Real_l(r) => Real_v(r)
    | L.Var_l(i) =>
	  (* lookup the variable's value in the dynamic environment *)
	  lookup(!dynamic_env,i)
    | L.Fn_l(e) =>
	  (* build a closure:  a pair of the code for the function,a
	   * and the current dynamic environment.  When the closure
	   * is applied to an argument, it will be evaluated in the
	   * context of this saved environment. *)
	  malloc_tuple [Code_v(e),!dynamic_env]
    | L.App_l(e1,e2) =>
	  (* apply a closure:  first, evaluate e1 and then save its value
	   * on the operand stack *)
	  let val _ = push (eval e1)
	      (* evaluate the argument *)
	      val v2 = eval e2
	      (* restore the closure *)
	      val v1 = pop()
	      (* get the code for the closure *)
	      val v1_code =
		  case get_ith(v1,1) of
		      Code_v e => e | _ => error("expecting code")
	      (* get the dynamic environment of the closure *)
	      val v1_env = get_ith(v1,2)
	      (* save our current dynamic environment on the stack *)
	      val _ = push (!dynamic_env)
	      (* add the argument to the beginning of the closure's
	       * environment, and then install the resulting new
	       * environment as the current dynamic environment. *)
	      val _ = dynamic_env := malloc_tuple[v2,v1_env]
	      (* evaluate the code of the closure in its extended env. *)
	      val result = eval v1_code
	  in
	      (* restore our saved dynamic environment *)
	      dynamic_env := pop();
	      (* return the result of the function call *)
	      result
	  end
    | L.Unop_l(L.Neg_l,e) =>
	  (case eval e of
	       Int_v(i) => Int_v(~i)
	     | Real_v(r) => Real_v(~r)
	     | _ => error("bad value for ~"))
    | L.Unop_l(L.Not_l,e) =>
	  (case eval e of
	       Int_v(0) => Int_v(1)
	     | Int_v(1) => Int_v(0)
	     | _ => error("bad value for not"))
    | L.Unop_l(L.Ptr_l,e) =>
	  (case eval e of
	       Ptr_v(_) => Int_v(1)
	     | _ => Int_v(0))
    | L.Binop_l(e1,b,e2) =>
             let
               val _ = push (eval e1)
               val v2 = eval e2
               val v1 = pop()
             in
               case (v1,b,v2) of
                 (Int_v i,L.Plus_l,Int_v j) => Int_v(i+j)
               | (Real_v i,L.Plus_l,Real_v j) => Real_v(i+j)
               | (Int_v i,L.Times_l,Int_v j) => Int_v(i*j)
               | (Real_v i,L.Times_l,Real_v j) => Real_v(i*j)
               | (Int_v i,L.Minus_l,Int_v j) => Int_v(i-j)
               | (Real_v i,L.Minus_l,Real_v j) => Real_v(i-j)
               | (Int_v i,L.GreaterThan_l,Int_v j) =>
		   if i > j then Int_v(1) else Int_v(0)
               | (Real_v i,L.GreaterThan_l,Real_v j) =>
		   if i > j then Int_v(1) else Int_v(0)
               | (Int_v i,L.LessThan_l,Int_v j) =>
		   if i < j then Int_v(1) else Int_v(0)
               | (Real_v i,L.LessThan_l,Real_v j) =>
		   if i < j then Int_v(1) else Int_v(0)
               | (Int_v i,L.Equal_l,Int_v j) =>
		   if i = j then Int_v(1) else Int_v(0)
               (* String Concatenation *)
               | (Ptr_v s1, L.Concat_l, Ptr_v s2) =>
                             let
                               val strListA = readString(s1)
                               val strListB = readString(s2)
                               (* chop leading 0's from second string *)
                               val newString = strListA@(tl(tl(strListB)))
                             in
                               malloc_tuple newString
                             end

               | (Ptr_v p1, L.Equal_l, Ptr_v p2) =>
                    (case get_ith(Ptr_v(p1),0) of
                      Tag_v(l) => if l >= 2 then
                                    let
                                      val strListA = 
                                        (map valToChar (readString(p1)))
                                      val strListB = 
                                        (map valToChar (readString(p2)))
                                    in
                                      if strListA = strListB then
                                        Int_v(1)
                                      else
                                        Int_v(0)
                                    end
                                  else
                                    if p1=p2 then Int_v(1) else Int_v(0)

                    | _ => error("tag expected"))


               | (_,_,_) => error("bad binop")

             end
    | L.Tuple_l(es) =>
	  (* evaluate the expressions and push them on the operand
	   * stack (in case a GC happens).  Then pop all of them off
	   * and allocate them in the memory, returning a pointer
	   * to the allocated memory as the result.
	   *)
	  let val _ = List.app (fn e => push (eval e)) es
	      val vs = rev (List.map (fn _ => pop()) es)
	  in
	      malloc_tuple vs
	  end
    | L.Ith_l(e1,e2) =>
	  let val _ = push (eval e1)
	      val v2 = eval e2
	      val v1 = pop()
	  in
	      case v1 of
		  Int_v(i) => get_ith(v2,i)
		| _ => error("bad value in Ith_l")
	  end
    | L.SetIth_l(e1,e2,e3) =>
          (push (eval e1);
           push (eval e2);
           push (eval e3);
           let
             val v3 = pop()
             val v2 = pop()
           in
             case pop() of
               Int_v(index) => (set_ith(v2, index, v3); malloc_tuple [])
             | _            => error("bad value in SetIth_l")

           end)
    | L.If_l(e1,e2,e3) =>
          let val _ = push(eval e1)
            val res = (case pop() of
                         Int_v(0) => e3
                       | Int_v(1) => e2
                       | _        => error("bad value in If_l"))
          in
            eval res
          end
    | L.Letrec_l(e1,e2) =>
	  (* Allocate a closure -- the code for the function (e1) and
	   * temporarily set the environment for the closure to null *)
	  let val v = malloc_tuple[Code_v e1,Int_v(0)]
	      (* save this closure on the stack (in case of GC) *)
	      val _ = push v
	      (* extend the current environment with the closure *)
	      val new_env = malloc_tuple[v,!dynamic_env]
	      val v = pop()
	      (* set the closures environment to the extended environment --
	       * notice that the closure refers to the environment, and the
	       * environment refers to the closure -- so there's a cyclic
	       * data structure to reflect the recursion. *)
	      val _ = set_ith(v,2,new_env);
	      (* set the current dynamic enviornment to the new extended env.*)
	      val _ = dynamic_env := new_env
	      (* evaluate the body of the letrec *)
	      val result = eval e2
	  in
	      (* restore the old dynamic env *)
	      dynamic_env := get_ith(!dynamic_env,2);
	      (* return the result *)
	      result
	  end
    | L.Error_l(s) => error(s)

  fun evaluate(e:L.lexp):value =
      let val v = eval e
      in
	  print "result = "; print_value v; print "\n";
	  if (!debug) then
	      (S.print_stack();
	       M.print_memory();
	       print "dynamic_env = "; print_value (!dynamic_env); print "\n")
	  else ();
	  v
      end

  fun evaluate_decl(e:L.lexp):value =
      let val v = evaluate(e)
      in
	  dynamic_env := malloc_tuple[v,!dynamic_env];
	  v
      end

  (* reset stack and memory to original state *)
  fun reset() =
     (M.reset();
      S.reset())

end
