structure MiniML :> sig

  val interpreter : unit -> unit

end = struct

  structure T = TypeCheck
  structure TC = TypeContext
  structure A = AbSyn
  structure LI = LambdaInterp
  structure C = Compile
  structure O = LambdaOpt
  structure CSE = CSEReplace

  (* The toplevel environment: keep a substitution for toplevel
   * declarations, and a type context for the type of the toplevel
   * identifiers
   *)
  type global_env = {type_context: TC.context, compiler_context:C.senv}

  val optimize   = ref false
  val cseReplace = ref false
  val constFold  = ref false
  val noisy      = ref true

  (* reconstruct a string from a list of tokens, inserting spaces *)
  fun reconstructString (sl:string list):string =
    ListFormat.fmt {final="",init="",fmt=fn x => x,sep=" "} sl

  (* the interpreter entry point *)
  fun interpreter ():unit =
    let

      (* execute a command *)
      fun interpretCommand (s:string,genv:global_env):unit =
        let

          (* load an expression from a file *)
          fun load (f:string):unit =
            let
              val instream = TextIO.openIn (f)
              val parsed = Parser.parse(instream)
              val _ = TextIO.closeIn(instream)
              val newEnv = interpretParse (parsed,genv)
            in
              interpreter_loop (newEnv)
            end

          (* display the parse tree of an expression *)
          fun show_parse (rest:string list):unit =
            let
              val s = reconstructString (rest)
            in
              (case Parser.parseString (s) of
                NONE => ()
              | SOME (A.Exp_t (e)) => (print (PrettyPrinter.ppExp (e));
                                       print "\n")
              | SOME(A.Decl_t (ds))=>
                  (print (foldr (fn (d,s)=>(PrettyPrinter.ppDecl d)^s)"" ds);
                   print "\n");
              interpreter_loop genv)
            end

          (* perform an underlying system call *)
          fun system (rest:string list):unit =
            let
              val s = reconstructString (rest)
            in
              OS.Process.system (s);
              interpreter_loop (genv)
            end

        in
          case Substring.getc (Substring.all (s)) of
            NONE => interpreter_loop (genv)
          | SOME (#":",ss) =>
              let
                val t = String.tokens Char.isSpace (Substring.string (ss))
              in
                case t of
                  ("l"::f::_) => load (f)
                | ("p"::rest) => show_parse (rest)
                | ("o"::rest) =>
                    (optimize := not(!optimize);
                     (if (!optimize) then print "Optimizer turned on\n" else
                        print "Optimizer turned off\n");
                        interpreter_loop(genv))
                | ("cse"::rest) =>
                    (cseReplace := not(!cseReplace);
                     (if (!cseReplace) then
                        print "CSE turned on\n"
                      else print "CSE turned off\n");
                        interpreter_loop(genv))
                | ("cf"::rest) =>
                    (constFold := not(!constFold);
                     (if (!constFold) then
                        print "Constant Folding turned on\n"
                      else print "Constant Folding turned off\n");
                        interpreter_loop(genv))
                | ("noisy"::rest) =>
                    (noisy := not(!noisy);
                     (if (!noisy) then
                        print "Noisy output turned on\n"
                      else print "Noisy output turned off\n");
                        interpreter_loop(genv))
                | ("s"::rest) => system (rest)
                | ("d"::rest) =>
                    (Memory.print_memory();
                     Stack.print_stack();
                     print("Dynamic Env = ");
                     Value.print_value(!LI.dynamic_env);
                     print("\n");
                     interpreter_loop(genv))
                | ("g"::rest) =>
                    (Stack.push(!LI.dynamic_env);
                     GC.gc 0; (* arg is amount of storage needed *)
                     LI.dynamic_env := (Stack.pop());
                     interpreter_loop(genv))
                | ("g2"::rest) =>
					(Stack.push(!LI.dynamic_env);
					GC.gc_2 0; (* arg is amount of storage needed *)
					LI.dynamic_env := (Stack.pop());
					interpreter_loop(genv))        
                | ("print"::ptr::rest) =>
                    (case Int.fromString(ptr) of
                       NONE => print "invalid pointer address\n\n"
                     | SOME(i) => print ("string: " ^ (LI.printString i)
                                         ^ "\n\n");
                         interpreter_loop (genv))
                | ("q"::_) => print "Bye.\n"
                | _ => (print "Command not recognized\n";
                        interpreter_loop (genv))
              end
          | _ => (print "Command not recognized\n";
                  interpreter_loop (genv))
        end

      (* if input is an expression or a declaration, parse
       * to obtain an abstract syntax tree
       *)
      and interpretParse (p:A.top_level option,genv:global_env) =
        let
          val {type_context,compiler_context} = genv
          fun comp_exp e =
            let
              val typ = T.tcheck (type_context,e)
                 handle Fail s => Error.static ("TypeChecker Failure: " ^ s)
              val lexp = C.comp_exp compiler_context e
              val _ = if (!noisy) then
                        (print "Compiled Expression:\n";
                         print "--------------------\n";
                         print (Lambda.ppLexp lexp);
                         print "\n\n")
                      else ()
              val lexp = if (!optimize) then O.optimize(lexp) else lexp
              val _ = if ((!noisy) andalso (!optimize)) then
                        (print "Optimized Expression:\n";
                         print "--------------------\n";
                         print (Lambda.ppLexp lexp);
                         print "\n\n")
                      else ()
              val lexp = if (!cseReplace) then
                           CSE.replace(lexp)
                         else lexp
              val _ = if ((!noisy) andalso (!cseReplace)) then
                        (print "CSE-Optimized Expression:\n";
                         print "--------------------\n";
                         print (Lambda.ppLexp lexp);
                         print "\n\n")
                      else ()
              val lexp = if (!constFold) then
                           ConstFold.const_fold(lexp)
                         else lexp
              val _ = if ((!noisy) andalso (!constFold)) then
                        (print "Constant Folded Expression:\n";
                         print "--------------------\n";
                         print (Lambda.ppLexp lexp);
                         print "\n\n")
                      else ()
            in
              ((if (not (!noisy)) then
                (print "Compiled Expression:\n";
                 print "--------------------\n";
                 print (Lambda.ppLexp lexp);
                 print "\n\n") else ());
              (typ,lexp))
            end
        in
          case p of
            NONE => genv
          | SOME(A.Decl_t([])) => genv
          (* evaluate an expression *)
          | SOME (A.Exp_t (exp)) =>
              let val (_, lexp) = comp_exp(exp) in LI.evaluate(lexp); genv end
          | SOME (A.Decl_t (A.Val_d(A.Id_p(x),e1)::ds)) =>
              (* for declarations:  val x = e, evaluate e and push
               * it into the interpreter's environment, and then
               * enter a binding into both the type-checker's and
               * compiler's context *)
              let val (typ,lexp) = comp_exp(e1)
              in
                (LI.evaluate_decl(lexp);
                 interpretParse
                 (SOME(A.Decl_t(ds)),
                       {type_context = TC.add_var(type_context, x, typ),
                         compiler_context = C.addvar compiler_context x}))
              end
          | SOME(A.Decl_t((d as A.Fun_d({name,arg,arg_typ,ret_typ},body))::ds))=>
              let val e = A.Let_e([d],A.Id_e(name))
                val (typ,lexp) = comp_exp(e)
              in
                (LI.evaluate_decl(lexp);
                 interpretParse
                 (SOME(A.Decl_t(ds)),
                  {type_context = TC.add_var(type_context, name, typ),
                   compiler_context = C.addvar compiler_context name}))
              end
          (* we don't deal with all of the other forms of declarations at
           * the top-level *)
          | SOME (A.Decl_t(_)) =>
              (print "Sorry, only val and fun top-level declarations ";
               print "are supported in this version of Mini-ML.\n";
               genv)
        end

      (* the main interpreter loop. It passes the toplevel env around *)
      and interpreter_loop (genv:global_env):unit =
        let fun processLine ():unit =
          (case TextIO.inputLine (TextIO.stdIn) of
             "" => ()
           | s => if (String.isPrefix ":" s) (* if it is a command *)
                    then interpretCommand (s,genv)
                  else let
                    val s = String.extract (s,0,SOME (size (s)-1))
                    val newEnv = interpretParse (Parser.parseString (s),genv)
                  in
                    interpreter_loop (newEnv)
                  end)
        in
          TextIO.output (TextIO.stdOut, "MiniML> ");
          TextIO.flushOut (TextIO.stdOut);
          processLine () handle Error.Error => interpreter_loop (genv)
        | e => (print "EXCEPTION: ";
                print (exnMessage (e));
                print "\n";
                interpreter_loop (genv))
        end
      val line = String.implode (List.tabulate (78,fn _ => #"-"))
    in
      SMLofNJ.Internals.GC.messages false;
      print (concat [line,"\nCS312 MiniML interpreter\n"]);
      LI.reset();
      interpreter_loop ({type_context=TC.empty_env,
                         compiler_context=C.empty_static_env})
    end

end


