signature COMPILE =
  sig
    type senv
    val empty_static_env : senv
    val addvar: senv -> AbSyn.id -> senv
    val comp_exp : senv -> AbSyn.exp -> Lambda.lexp
  end


structure Compile :> COMPILE =
  struct
    open AbSyn
    open Lambda

    (* Static environments:  we map each variable to an integer
     * corresponding to the lexical nesting depth.  The nearest
     * enclosing bound variable has index 0, the next 1, etc.
     *)
    type senv = id -> int

    exception UnboundVariable of string
    exception InternalError of string

    (* Given a list of fields (id*typ) and a field name, return
     * the index of the corresponding field.  Used for records.
     *)
    fun find(name:string,fs:(id*'a)list):int =
      case fs of
        [] => raise InternalError("find field lookup field")
      | (id,_)::rest => if (name = id) then 1 else 1+find (name,rest)

    (* Map a nullary data constructor to an integer value. *)
    fun nullary_datacon_tag x =
        case x of
            "NONE" => 0
          | "Nil" => 0
          | "true" => 1
          | "false" => 0
          | _ => raise InternalError("unknown constructor "^x)

    (* map a value-carrying data constructor to an integer value *)
    fun value_datacon_tag x =
        case x of
            "Cons" => 1
          | "SOME" => 1
          | _ => raise InternalError("unknown constructor "^x)

    (* sort for records *)
    fun field_sort(fs) =
        ListMergeSort.sort (fn ((x:id,_),(y:id,_)) => x > y) fs

    (* An empty static environment -- no variables are bound *)
    fun empty_static_env x =
        (print x; print " is unbound!\n"; raise UnboundVariable(x))

    (* Add a variable to the static environment -- map it to 0
     * and add one to the indices of all of the other variables. *)
    fun addvar (senv:senv) (x: id) : senv =
        fn (y:id) => if (x = y) then 0 else 1 + senv(y)

    (* Used to generate "fresh" variable names during pattern match
     * compilation. *)
    local
        val counter = ref 0
    in
        fun gensym() =
            let val c = Int.toString(!counter)
            in
                counter := (!counter) + 1;
                "%temp"^c
            end
    end

    (* compile a MiniML unary operation to a Lambda unary operation *)
    fun comp_unop (u:unop) : lunop =
        case u of
            Neg => Neg_l
          | Not => Not_l

    (* compile a MiniML binary operation to a Lambda binary operation *)
    fun comp_binop (l1: lexp, b:binop, l2:lexp) : lexp =
        case b of
            Plus => Binop_l(l1, Plus_l, l2)
          | Times => Binop_l(l1,Times_l, l2)
          | Minus => Binop_l(l1, Minus_l, l2)
          | Equal => Binop_l(l1,Equal_l, l2)
          | Concat => Binop_l(l1,Concat_l, l2)
          | GreaterThan => Binop_l(l1, GreaterThan_l, l2)
          | LessThan => Binop_l(l1, LessThan_l, l2)
          | GreaterThanEq => Unop_l(Not_l,(Binop_l(l1,LessThan_l,l2)))
          | LessThanEq => Unop_l(Not_l,(Binop_l(l1,GreaterThan_l,l2)))

    (* compile a MiniML expression to a Lambda expression *)
    fun comp_exp (senv : senv) (e:exp) : lexp =
        case e of
            Const_e(Int_c i) => Int_l(i)
          | Const_e(Real_c r) => Real_l(r)
          | Const_e(String_c s) =>
              (* strings are represented as tuples of integers *)
              (* Old representation commented out here. New representation
               below that. This allows us to do pointer equality.
               Tuple_l(map (Int_l o ord) (String.explode s)) *)
              Tuple_l(map Int_l (0::0::(map ord (String.explode s))))
          | Const_e(Char_c c) =>
              (* characters are reprsented as integers *)
              Int_l(ord(c))
          | Id_e x =>
              (* look up the index for the variable in the static env *)
              Var_l (senv x)
          | Fn_e(x,t,e) =>
              (* add x as the variable at index 0 and compile the body
               * of the function. *)
              Fn_l(comp_exp (addvar senv x) e)
          | App_e(e1,e2) => App_l(comp_exp senv e1, comp_exp senv e2)
          | Unop_e(u,e) => Unop_l(comp_unop u,comp_exp senv e)
          | Binop_e(e1,b,e2) =>
              comp_binop(comp_exp senv e1, b, comp_exp senv e2)
          | Tuple_e(es) => Tuple_l(List.map (comp_exp senv) es)
          | Ith_e(i,e) => Ith_l(Int_l(i),comp_exp senv e)
          | Record_e(ides) =>
              Tuple_l(List.map (fn (_,e) => comp_exp senv e)
                      (field_sort ides))
          (* we represent records as tuples with the components
           * sorted by field name *)
          | Field_e(id,e,toptr) =>
              (case !toptr of
                 SOME (Record_t fields) =>
                   let val index = find(id,field_sort(fields))
                   in Ith_l(Int_l(index),comp_exp senv e) end
               | _ => raise (InternalError("type-checker not run")))
          | DataCon_e(x,NONE) =>
              (* Nullary data constructors are compiled to integers *)
              Int_l(nullary_datacon_tag(x))
          | DataCon_e(x,SOME e) =>
              (* Value-carrying data constructors are compiled to
               * a tuple with an integer in the first slot, and the
               * argument (e) in the second slot.  So, for instance,
               * Cons(42,Nil) gets compiled to (1,(42,0))
               *)
              Tuple_l[Int_l(value_datacon_tag(x)),comp_exp senv e]
          | Case_e(e,cases) =>
              let val le = comp_exp senv e
                (* generate a fresh variable for expression e *)
                val x = gensym()
                (* compile the case patterns and sub-expressions in
                 * such a way that they test the variable x *)
                val testle = comp_cases (addvar senv x) x cases
              (* essentially (fn x => patterns)(e) *)
              in App_l(Fn_l testle,le) end
          | Let_e(Val_d(p,e1)::d,e2) =>
              (* treat this as a special case of case expressions *)
              let val inner = Let_e(d,e2)
              in comp_exp senv (Case_e(e1,[(p,inner)])) end
          | Let_e(Fun_d({name,arg,...},body)::d,e) =>
              (* compile down to a letrec expression *)
              let val inner = Let_e(d,e)
              in
                Letrec_l(comp_exp (addvar (addvar senv name) arg) body,
                         comp_exp (addvar senv name) inner)
              end
          | Let_e([],e2) => comp_exp senv e2
          | Deref_e(e) => Ith_l(Int_l(1),comp_exp senv e)
          | Assign_e(e1,e2) => SetIth_l(Int_l(1),comp_exp senv e1,
                                        comp_exp senv e2)
          | Ref_e(e) => Tuple_l [comp_exp senv e]

    (* Compile a pattern:  senv is the current static environment,
     * root is the expression that we're testing and deconstructing,
     * ok generates the code for when the pattern test succeeds,
     * and fail generates the code for when the pattern test fails.
     * Note that ok and fail take a static environment as an argument
     * because the pattern may add variables to the environment. *)
    and pat_compile (p:pat) (senv:senv) (root:lexp) (ok:senv->lexp)
        (fail:senv->lexp) : lexp =
        case p of
            Wild_p =>
                (* wild patterns alway succeed *)
                ok senv
          | Id_p(x) =>
                (* add the root to the static environment as x *)
                App_l(Fn_l(ok (addvar senv x)),root)
          | Const_p(i) =>
                (* generate code to test that root is equal to i *)
                If_l(Binop_l(root,Equal_l,Int_l i),ok senv,fail senv)
          | DataCon_p(x,NONE) =>
                (* generate code to test that the root is (a) not a pointer
                 * and (b) is equal to the appropriate integer that represents
                 * the data constructor. *)
                If_l(And_l(Unop_l(Not_l,Unop_l(Ptr_l,root)),
                           Binop_l(root,Equal_l,
                                   Int_l(nullary_datacon_tag(x)))),
                     ok senv,fail senv)
          | DataCon_p(x,SOME(p)) =>
                (* generate code to test that the root is (a) a pointer
                 * to a tuple, (b) the first component of the tuple is
                 * the appropriate integer that represents the data
                 * constructor, and (c) that the sub-pattern matches
                 * the second components of the tuple. *)
                If_l(And_l(Unop_l(Ptr_l, root),
                           Binop_l(Ith_l(Int_l(1),root),Equal_l,
                                   Int_l(value_datacon_tag(x)))),
                     pat_compile p senv (Ith_l(Int_l(2),root)) ok fail,
                     fail senv)
          | Tuple_p(ps) =>
                (* generate code to test each pattern from left-to-right
                 * against the root. *)
                let val x = gensym()
                    val senv = addvar senv x
                    fun loop (ps,i) senv =
                        case ps of
                            [] => ok senv
                          | p::rest =>
                                pat_compile p senv (Ith_l(Int_l(i),
                                                          Var_l(senv x)))
                                (loop (rest,i+1)) fail
                in
                    App_l(Fn_l(loop(ps,1) senv),root)
                end
          | Record_p(idps) =>
                (* sort the record and then compile it the same way
                 * we would compile a tuple pattern match. *)
                let val sorted_idps = field_sort idps
                    val ps = map (fn (_,p) => p) sorted_idps
                in
                    pat_compile (Tuple_p(ps)) senv root ok fail
                end

    (* compile a list of cases "p1=>e1 | p2=>e2 | ... | pn=>en"
     * into an appropriate test expression.  senv is the current
     * static environment, x is an identifier bound to the value
     * being tested, and cases is the list of patterns and expressions
     * to be compiled. *)
    and comp_cases (senv: id->int) (x:id) (cases : (pat*exp) list) : lexp =
        case cases of
            [] =>
                (* generate code for an inexhaustive match *)
                Error_l("Match problem")
          | (p,e)::rest =>
                (* compile pattern p -- if the resulting code
                 * succeeds in the pattern match, then evaluate
                 * e, otherwise continue with the next pattern
                 * in the list. *)
                pat_compile p senv (Var_l(senv x))
                (fn senv => comp_exp senv e)
                (fn senv => comp_cases senv x rest)

  end

