signature RSA = sig

  (* alias B to RSABigNums structure *)
  structure B : BIGNUMS = RSABigNums

  (* A key is the modulus and exponent pair (n,e) for a public key and
   * (n,d) for a private key. Because we require large keys we use bignum
   * types for both the modulus and the exponent. *)
  type key = {modulus: B.bignum, exponent: B.bignum}

  (* A key pair consists of two keys, the public key (n,e) and the
   * private key (n,d). It should be the case that the keys form a
   * compatible public / private key pair generated by a function such as
   * generateRSAKeyPair. *)
  type key_pair = {public: key, private: key}

  (* A signed message is a message (a bignum list) and a signature (a
   * bignum). *)
  type signed_message = {msg: B.bignum list, sign: B.bignum}

  (* Effects: printNumList prints out the numbers in a bignum list *)
  val printNumList : B.bignum list -> unit

  (* stringToNumList(s) is a bignum list in which each number in the    
   * list is between 0 and 2^28 and corresponds to a four-character     
   * substring of s. This function produces a bignum list that can be   
   * converted back to s via numListToString. *)                        
  val stringToNumList : string -> B.bignum list

  (* numListToString is the inverse of stringToNumList. The input       
   * is a list of bignums between 0 and 2^28 and is mapped to the       
   * concatenation of the strings that each number represents. *)       
  val numListToString : B.bignum list -> string

  (* generateRSAKeyPair(r) is a public/private key pair
   * based on primes of size 2^r or greater *)
  val generateRSAKeyPair : int -> key_pair

  (* encrypt(s,k) is the encryption of the string s *)
  val encrypt : string * key -> B.bignum list

  (* decrypt(L,k) is the decryption of L.
     decrypt(encrypt(s,k),k') = s if k and k' are a key pair. *)
  val decrypt : B.bignum list * key -> string

  (* encryptList(L,k) is the RSA encryption of L, assuming L
   * represents an unencrypted string. *)
  val encryptList : B.bignum list * key -> B.bignum list

  (* decryptList(L,k) is the RSA decryption of L, assuming L
   * represents the encryption of a string. That is,
   * decryptList(encryptList(L,kpub),kpriv) = L
   * where {kpub, kpriv} is a public/private key pair. *)
  val decryptList : B.bignum list * key -> B.bignum list

  (* encryptAndSign(m,ksend,krec) is the signed, encrypted version of m.
   * It is signed with the private key ksend and encrypted with the
   * public key krec. *)
  val encryptAndSign : string * key * key -> signed_message

  (* authenticateAndDecrypt(m,ksend,krec) is the decryption of
   * the signed message m. ksend is the public key of the sender and
   * krec is the private key of the receiver.
   * It is the inverse of encryptAndSign:
   * authenticateAndDecrypt(encryptAndSign(m,kspriv,krpub),kspub,krpriv) = m
   * where {kspub, kspriv} and {krpub, krpriv} are public/private key pairs.
   * Checks: m was signed by the sender. *)
  val authenticateAndDecrypt : signed_message * key * key -> string

  (* smallestDivisor(N) is the smallest integer divisor of N that is
   * greater than 1.
   * Requires: N > 1 *)                   
  val smallestDivisor : B.bignum -> B.bignum

  (* crack(pubkey) is the private key corresponding to pubkey.
   * May take a very long time if the key is large. *)
  val crack : key -> key
end
