structure RSABigNums :> BIGNUMS  = struct
  structure I = IntInf

  type bignum = I.int

  val toString = I.toString
  val fromString = I.fromString
  fun toInt (i:bignum):int option = SOME (I.toInt (i)) handle _ => NONE
  val fromInt = I.fromInt

  val negate = I.~

  fun equal (i1:bignum, i2:bignum):bool = (i1=i2)
  val less = I.<
  val compare = I.compare

  val plus = I.+
  val minus = I.-
  val times = I.*
  val divmod = I.divmod
end

structure RSASupport1 = struct
  structure B = RSABigNums

  type bignum = B.bignum
  type key = {modulus: bignum, exponent: bignum}
  type key_pair = {public: key, private: key}
  type signed_message = {msg: bignum list, sign : bignum}

  val b = B.fromInt            (* Use this to convert int to bignum *)
  val zero : bignum = b 0
  val one : bignum = b 1
  val two : bignum = b 2
  val three : bignum = b 3

  (* Function to select at random from a range. You are not
   * responsible for this function, which uses features of
   * SML you have not yet seen *)
  local
    val seed = Random.rand (12345,67890)
  in
    fun random r = let
      val r' = case (B.toInt r)
                 of NONE => valOf (Int.maxInt)
                  | SOME i => i
    in
      b (Random.randRange (0,r') seed)
    end
  end

  (* Rebind the math operators so they use bignums instead of int.
   * This means that to do math on integers, we must reference the
   * Int structure explicitly. *)
  val op + = B.plus
  val op - = B.minus
  val op * = B.times
  val op mod = #2 o B.divmod
  val op div = #1 o B.divmod

  fun even (b:bignum):bool = B.equal((b mod two), zero)
  fun square (b:bignum):bignum = b*b
  fun exp (b:bignum,e:bignum):bignum =
    if B.equal(e, zero) then one else b * exp(b, e-one)

  (* The functions that follow are explained in the problem set *)
  fun expmod (b:bignum, e:bignum, m:bignum):bignum =
    if B.equal(e, zero) then one
    else if even e then
           square(expmod(b, e div two, m)) mod m
         else (b * expmod(b, e-one, m)) mod m

  fun transform (number:bignum, {modulus,exponent}:key):bignum =
    expmod (number,exponent,modulus)

  fun compress (numlist:bignum list):bignum = let
    val base : bignum = exp (b 2, b 18)
    fun addLoop (l:bignum list):bignum =
      case l of
        [] => zero
      | x::xs => x + addLoop xs
  in
    (addLoop numlist) mod base
  end

  fun fermatTest (n:bignum,a:bignum):bool =
    B.equal(expmod (a,n,n), a)

  fun fastPrime (n:bignum):bool =
    fermatTest (n,b 2) andalso
    fermatTest (n,b 3) andalso
    fermatTest (n,b 5) andalso
    fermatTest (n,b 7)

  fun searchForPrime (guess:bignum):bignum =
    if fastPrime guess then guess
    else searchForPrime (guess + two)

  fun choosePrime (smallest:bignum, range:bignum):bignum = let
    val start = smallest + (random range)
  in
    searchForPrime (if even start then start+one else start)
  end

  fun euclid (m:bignum,n:bignum) : (bignum * bignum * bignum) =
    if B.equal(n, zero) then (one, zero, m)
    else let
      val q = m div n
      val r = m mod n
      val (u,v,g) = euclid (n,r)
    in
      (v, u-(q*v), g)
    end

  fun selectKeys (m:bignum):(bignum * bignum) = let
    val e = random m
    val (u,v,g) = euclid (e,m)
  in
    if B.equal(g, one) then (e, u mod m)
    else selectKeys m
  end

  fun generateRSAKeyPair (r':int):key_pair = let
    val r = b r'
    val size = exp (two, r)
    val p = choosePrime (size,size)
    val q = choosePrime (size,size)
  in
    if B.equal(p, q) then generateRSAKeyPair r'
    else let
      val n = p * q
      val m = (p-one) * (q-one)
      val (e,d) = selectKeys m
      val a : key = {modulus=n,exponent=e}
      val b : key = {modulus=n,exponent=d}
    in
      {public=a, private=b}
    end
  end

  fun stringToNumList (s:string):bignum list =
    let
      val op + = Int.+   (* Here we want to do integer operations. *)
      val op * = Int.*
      fun charsToInt (x:char,y:char):bignum = b (ord x + 128 * ord y)
      val charList = explode s
      fun pairUp (x:char list):(char * char) list =
        case x of
          [] => []
        | [x] => [(x,#" ")]
        | x::y::xs => (x,y)::pairUp xs
    in
      map charsToInt (pairUp charList)
    end

  fun numListToString (l:bignum list):string = let
    val op mod = Int.mod   (* Here we want to do integer operations. *)
    val op div = Int.div
    fun intToChars (b:bignum):(char * char) =
      let
        val i = (case B.toInt b of
                   NONE => raise Fail "Internal Error"
                 | SOME x => x)
        val a = i mod 128
        val b = (i div 128) mod 128
      in
        (chr a, chr b)
      end
    fun f (v:bignum, r:char list):char list =
      let
        val (x,y) = intToChars v
      in
        x::y::r
      end
  in
    implode (foldr f [] l)
  end

  fun printNumList (l:bignum list):unit = let
    fun toString (l:bignum list):string =
      case l of
        [] => ""
      | [x] => B.toString (x)
      | x::xs => String.concat [B.toString x, ",", toString xs]
  in
    print (String.concat ["[", toString l, "]\n"])
  end

  fun encryptList (numList:bignum list, k:key):bignum list =
    case numList of
      [] => []
    | x::xs =>
        let
          val xs' = encryptList (xs,k)
          val n = (case xs' of
                     [] => x
                   | y::ys => (x + y) mod (#modulus k))
          val x' = transform (n, k)
        in
          x'::xs'
        end

  fun encrypt (s:string,k:key):bignum list =
    encryptList (stringToNumList s, k)
end