(* Do not edit outside the indicated areas *)

structure BigNums :> BIGNUMS = struct

  type bignum = { neg: bool, coeffs: int list }

  structure P1 = Problem1
  structure P2 = Problem2
  structure P3 = Problem3
  structure OPTPROB = OptionalProblem

  (* Abstraction function:
   *
   * Given a bignum with coeffs = [a0, a1, ..., an], the magnitude
   * of the bignumber is
   *
   *     a0 + a1*base + a2*base^2 + ... + an*base^n
   *
   * and the sign of the number is given by neg.
   *
   * For example, {neg=true, coeff=[20,1]} represents 1020 in base 1000.
   *
   * Representation invariant: see problem2.sml
   *)

  val base = BigNumsSupport1.base

(* Some useful bignumbers *)
  val toString = P1.toString
  val equal = P2.equal
  val less = P2.less
  val compare = P2.compare
  val negate = P3.negate
  val plus = P3.plus
  val times = P3.times
  val divmod = OPTPROB.divmod

  val zero={neg=false, coeffs=[]}
  val one={neg=false, coeffs=[1]}
  val two={neg=false, coeffs=[2]}
  val minusOne={neg=true, coeffs=[1]}
  val baseBig={neg=false, coeffs=[0,1]}

  (* Make sure ten satisfies RI if base is changed *)
  val ten = {neg=false, coeffs=[10]}

  fun minus(a: bignum, b: bignum):bignum = plus(a, negate(b))

  fun toInt ({neg,coeffs}:bignum):int option =
    let
      fun loop l =
        (case l of
           [] => 0
         | (x::xs) => x + (base * loop (xs)))
    in
      SOME ((if neg then ~1 else 1) * loop (coeffs)) handle _ => NONE
    end

  fun fromInt (n:int):bignum =
    let
      fun loop (n) = if (n=0) then []
                     else (n mod base)::loop (n div base)
    in
      {neg=(n<0), coeffs=loop (Int.abs (n))}
    end


  val fromString:(string -> bignum option) = let
    fun digit c = if c = #"0"
                    then zero
                  else {neg=false,coeffs=[Char.ord (c) - Char.ord #"0"]}
    fun scan getc cs = let
      fun loop (comb,cs,curr) =
        case (getc (cs))
          of NONE => SOME (curr,cs)
           | SOME (c,cs') => (if (Char.isDigit (c)) then
                                loop (comb,cs',comb(times(curr,ten), digit(c)))
                              else SOME (curr,cs))
    in
      case (getc (StringCvt.skipWS getc cs))
        of NONE => NONE
         | SOME (c,cs') => (if (Char.isDigit (c)) then
                              loop (plus,cs',digit (c))
                            else if (c = #"~") then loop (minus, cs', zero)
                                 else NONE)
    end
  in
    StringCvt.scanString scan
  end

end
