
structure PrintDebug :> sig
  val printExp:       AbstractSyntax.exp  * int                        -> string
  val printEnv:       Environment.env     * int                        -> string
  val printValue:     ((Environment.value * AbstractSyntax.typ) * int) -> string
  val prettyPrintAST: AbstractSyntax.top_level option * int            -> string
  val printT:         AbstractSyntax.typ  * int                        -> string
end
= struct

  open AbstractSyntax
  open Environment

  val err = Error.runtime

  val step = 4

  (* Makes a string of n spaces. *)
  fun nspaces (n: int): string =
  let
    fun helper(n: int, c: int): string =
      case (n, c) of
        (0, _) => ""
      | (_, 1) => " " ^ helper(n - 1, step)
      | (_, s) => (if s = step then "." else " ") ^ helper(n - 1, c - 1)
  in
    helper(n, step)
  end

  (* Prints a list of expressions, types, or declarations. *)
  fun printLst (lst: 'a list, prt: 'a * int -> string, n: int): string =
    let
      val s = nspaces n
      fun helper (lst: 'a list): string =
        case lst of
          []   => ""
        | h::t => prt(h, n + step) ^ helper t
    in
      helper lst
    end

  (* Prints a list of formal arguments declarations. *)
  fun printArgs(dlst: (id * typ) list, n: int): string =
    let
      val s  = nspaces(n)
      val s2 = nspaces(n + step);
      val s3 = nspaces(n + 2 * step);
    in
      foldl (fn((i,t),str) => str ^ s2 ^ "Arg(\n"
                                  ^ s3 ^ i ^ "\n"
                                  ^ printT(t, n + 2 * step)
                                  ^ s2^ ")\n") "" dlst
    end

  (* Prints a declaration. *)
  and printDecl (d: decl, n: int): string =
   let
     val s  = nspaces(n)
     val s2 = nspaces(n + step);
   in
     case d of
       Val_d(id,tp,exp) => s ^ "Val_d(\n"
                       ^ s2 ^ id ^ "\n"
                       ^ printT(tp, n + step)
                       ^ printExp(exp, n + step)
                       ^ s ^ ")\n"
     | Fun_d({name, args, ret_typ}, exp) =>
                       s ^ "Fun_d(\n"
                       ^ s2 ^ "name = " ^ name ^ "\n"
                       ^ s2 ^ "args = \n"
                       ^ s2 ^ "[\n"
                       ^ printArgs(args, n)
                       ^ s2 ^ "]\n"
                       ^ s2 ^ "return type = \n"
                       ^ printT(ret_typ, n + 2 * step)
                       ^ s2 ^ "),\n"
                       ^ s2 ^ "expression = \n"
                       ^ printExp(exp, n + step)
    end

  (* Prints an expression. *)
  and printExp (e: exp, n: int) =
    let
      val s  = nspaces(n)
    in
      case e of
        Int_c    x    => s ^ "Int_c("  ^ Int.toString  x  ^ ")\n"
      | Real_c   x    => s ^ "Real_c(" ^ Real.toString x  ^ ")\n"
      | Bool_c   x    => s ^ "Bool_c(" ^ Bool.toString x  ^ ")\n"
      | Char_c   x    => s ^ "Char_c(" ^ Char.toString x  ^ ")\n"
      | String_c x    => s ^ "String_c(\"" ^ x ^ "\")\n"
      | Id_e     x    => s ^ "Id_e(" ^ x ^ ")\n"
      | If_e(c, t, f) => s ^ "If_e(\n"
                       ^ printExp(c, n + step)
                       ^ printExp(t, n + step)
                       ^ printExp(f, n + step)
                       ^ s ^ ")\n"
      | Let_e(dl, e)  => s ^ "Let_e(\n"
                       ^ printLst(dl, printDecl, n)
                       ^ printExp(e, n + step)
                       ^ s ^ ")\n"
      | Fn_e(args, t, e) =>
                         s ^ "Fn_e(\n"
                       ^ printArgs(args, n)
                       ^ printT(t, n + step)
                       ^ printExp(e, n + step)
                       ^ s ^ ")\n"
      | Apply_e(e1, e2)=> s ^ "Apply_e(\n"
                       ^ printExp(e1, n + step)
                       ^ printExp(e2, n + step)
                       ^ s ^ ")\n"
      | Unop_e(uo, e) => s ^ "Unop_e(\n"
                       ^ nspaces(n + step)
                           ^ (case uo of
                                Neg   => "Neg"
                              | Not   => "Not"
                              | Ref   => "Ref"
                              | Deref => "Deref")
                           ^ "\n"
                       ^ printExp(e, n + step)
                       ^ s ^ ")\n"
      | Binop_e(e1, bo, e2) => s ^ "Binop_e(\n"
                       ^ nspaces(n + step)
                           ^ (case bo of
                                Plus       => "Plus"
                              | Times      => "Times"
                              | Minus      => "Minus"
                              | Mod        => "Mod"
                              | Div        => "Div"
                              | Slash      => "Slash"

                              | Equal      => "Equal"
                              | Greater    => "Greater"
                              | Less       => "Less"
                              | GreaterEq  => "GreaterEq"
                              | LessEq     => "LessEq"

                              | Cons       => "Cons"
                              | Append     => "Append"

                              | AndAlso    => "AndAlso"
                              | OrElse     => "OrElse"

                              | Concat     => "Concat"

                              | Assign     => "Assign")
                           ^ "\n"
                       ^ printExp(e1, n + step)
                       ^ printExp(e2, n + step)
                       ^ s ^ ")\n"
      | Tuple_e el    => s ^ "Tuple_e(\n"
                       ^ printLst(el, printExp, n)
                       ^ s ^ ")\n"
      | Ith_e(i, e)   => s ^ "Ith_e(\n"
                       ^ nspaces(n + step) ^ Int.toString(i) ^ "\n"
                       ^ printExp(e, n + step)
                       ^ s ^ ")\n"
      | List_e  el    => s ^ "List_e(\n"
                       ^ printLst(el, printExp, n)
                        ^ s ^ ")\n"
      end

  (* Prints a type's abstract syntax tree. *)
  and printT (t: typ, n: int) =
    let
      val s = nspaces n
    in
      case t of
        Int_t        => s ^ "Int_t\n"
      | Real_t       => s ^ "Real_t\n"
      | Bool_t       => s ^ "Bool_t\n"
      | Char_t       => s ^ "Char_t\n"
      | String_t     => s ^ "String_t\n"
      | Tuple_t tl   => s ^ "Tuple_t(\n"
                      ^ printLst(tl, printT, n)
                      ^ s ^ ")\n"
      | List_t t     => s ^ "List_t(\n"
                      ^ printT(t, n + step)
                      ^ s ^ ")\n"
      | Ref_t t      => s ^ "Ref_t(\n"
                      ^ printT(t, n + step)
                      ^ s ^ ")\n"
      | Fn_t(t1, t2) => s ^ "Fn_t(\n"
                      ^ printT(t1, n + step)
                      ^ printT(t2, n + step)
                      ^ s ^ ")\n"
      | Undefined_t  => s ^ "Undef_t\n"
    end


  (* Prints an abstract syntax tree. *)
  fun prettyPrintAST (ast, n): string =
  let
    val s = nspaces n
  in
    case ast of
      NONE             => raise Fail "prettyPrintAST => internal error!"
    | SOME(Decl_t lst) => s ^ "Decl_t(\n" ^ printLst(lst,printDecl,n)^s^")\n"
    | SOME(Exp_t exp)  => s ^ "Exp_t(\n" ^ printExp(exp, n + step) ^ s ^ ")\n"
  end


  (* Prints type ASTs in linear form. *)
  fun printTypeLst (lst: typ list): string =
    case lst of
      []   => ""
    | [t]  => printType t
    | h::t => printType h  ^ " * " ^ printTypeLst t

  and printType (t: typ): string =
    let
      (* We need the helper to non-redundantly paranthesize types. *)
      fun helper (t: typ, parans: bool):  string =
        case t of
          Int_t        => "int"
        | Real_t       => "real"
        | Bool_t       => "bool"
        | Char_t       => "char"
        | String_t     => "string"
        | Tuple_t tl   => (case tl of
                             []  => "unit"
                           | [x] => helper(x, parans)
                           | _   => "(" ^ printTypeLst tl ^ ")")
        | List_t tp    => helper(tp, true) ^ " list"
        | Ref_t t      => helper(t, true)  ^ " ref"
        | Fn_t(t1, t2) => if parans
                          then "(" ^ helper(t1, false) ^ " -> "
                                   ^ helper(t2, false) ^ ")"
                          else helper(t1, false) ^ " -> " ^ helper(t2, false)
        | Undefined_t  => "undefined"
    in
      helper(t, false)
    end

 (* Prints values. *)
 fun  printValue ((v, t): value * typ, n: int): string =
 let
   val s    = nspaces(n)
   val s2   = nspaces(n + step)
   fun printv (v: value, n: int): string =
   let
      val pl = foldl (fn (v, str) => str ^ ", " ^ (printv(v, n + step)))
   in
     case v of
      Int_v  i         => s ^ Int.toString i
    | Real_v r         => s ^ Real.toString r
    | Bool_v b         => s ^ (if b then "true" else "false")
    | Char_v c         => s ^ ("#\"" ^ Char.toString c ^ "\"")
    | String_v st      => s ^ ("\"" ^ st ^ "\"")
    | Tuple_v  []      => s ^ "()"
    | Tuple_v  (v::vs) => s ^ "("
                        ^ (pl (printv(v, n + step)) vs)
                        ^ s ^ ")"
    | List_v   []      => s ^ "[]"
    | List_v   (v::vs) => s ^ "["
                        ^ (pl (printv(v, n + step)) vs)
                        ^ "]"
    | Fn_v(a, e, b, nm)=> s ^ "closure = Fn_v(\n"
                        ^ s ^"  args = ("^(foldl(fn(s,a)=>a^" "^s) " " a)^" )\n"
                        ^ s ^ "  environment = (\n"
                        ^ printEnv(e, n + step)
                        ^ s ^ "  )\n"
                        ^ s ^ "  body = \n"
                        ^ prettyPrintAST(SOME(Exp_t b), n + step)
                        ^ s ^ "  name = " ^ (case nm of
                                              NONE =>"none (anonymous closure)"
                                            | SOME(nm2) => nm2 ) ^"\n"
                        ^ s ^ ")"
    | Predef_v st      => s ^ "predefined_function(" ^ st ^ ")"
    | SpecForm_v st    => s ^ "special form(" ^ st ^ ")"
    | Thunk_v(e, en)   => s ^ "thunk = (\n"
                        ^ s ^ "  body = \n"
                        ^ prettyPrintAST(SOME(Exp_t e), n + 2 * step)
                        ^ s ^ "  environment = (\n"
                        ^ printEnv(en, n + 2 * step)
                        ^ s2 ^ "  )\n"
                        ^ s  ^ ")"
    | Dyn_v(e)         => s ^ "dynamic variable = (\n"
                        ^ s ^ "  body = \n"
                        ^ prettyPrintAST(SOME(Exp_t e), n + 2 * step)
                        ^ s  ^ ")"


   end
 in
   printv(v, n) ^ ": " ^ printType(t) ^ "\n"
 end

 (* Prints (functional environment. *)
 and printEnv (Env e, n: int): string =
 let
   val s  = nspaces(n)
   val s2 = nspaces(n + step)
   fun helper (e: (string * value * typ) list): string =
   case e of
     []             => ""
   | (id, v, t)::tl => s ^ "(\n" ^ s2 ^ "name = " ^ id ^ "\n"
                     ^ printValue((v, t), n + step) ^ s ^ ")\n" ^ helper tl
 in
   helper(e)
 end

end
