(*
  The main loop of the MiniML interpreter.  To start the interpreter, use
  Interpreter.run ();
*)
structure Interpreter :> sig
  val run: unit -> unit
end

= struct

  open AbstractSyntax
  open Environment
  open Evaluator
  open Parser
  open PrintDebug

  val prompt = "Mini-ML >> "
  val help =
  "Summary of commands:\n"                                                   ^
  ":e toggles environment printing after evaluating declarations\n"            ^
  ":h prints this message\n"                                                   ^
  ":p prints the AST that corresponds to a Mini-SML expression\n"              ^
  ":q terminates program execution\n\n"                                        ^
  "If a line starts with a character other than ':', the text is considered\n"^
  "a Mini-SML expression. The result of its evaluation, if successful, will\n" ^
  "be printed.\n"

  fun loop (en: env, prenv: bool):unit =
    let
      val _ = print prompt
      val inLine = TextIO.inputLine TextIO.stdIn
      fun badControl () = print "Bad control sequence.\n"
    in
      if inLine = "" then loop(en, prenv)
      else if String.sub (inLine, 0) = #":" then
        if String.size inLine < 2 then (badControl (); loop(en, prenv))
        else (case String.sub (inLine, 1) of
                #"p" => (print (prettyPrintAST
                                (parseString
                                 (String.extract (inLine, 2, NONE)), 0));
                         loop(en, prenv))
              | #"q" => ()
              | #"h" => (print(help); loop(en, prenv))
              | #"e" => loop(en, not prenv)
              | _    => (badControl (); loop(en, prenv)))
      else
        let
          val t = parseString inLine
        in
          (case t of
             NONE             => loop(en, prenv)
           | SOME(Exp_t ex)   => (print(printValue(
                                          forceValue(evaluate (ex, en)),
                                          0));
                                  loop(en, prenv))
           | SOME(Decl_t dlist) => let
                                     val en = evaluateDeclare(dlist, en)
                                   in
                                     (if prenv then print(printEnv(en, 0))
                                               else ();
                                      loop(en, prenv))
                                   end)
             handle Error.Error => loop(en, prenv)
        end
    end

  fun run ():unit =
    (print "--- CS312 MiniML interpreter ---\n";
     loop(top_level, false))

end