(*
  Mappings from identifiers to values.
*)
structure Environment = struct

  open AbstractSyntax

  datatype value = Int_v      of int
                 | Real_v     of real
                 | Bool_v     of bool
                 | Char_v     of char
                 | String_v   of string
                 | Tuple_v    of value list
                 | List_v     of value list
                 | Fn_v       of (  string list    (* arguments            *)
                                  * env            (* environment          *)
                                  * exp            (* body                 *)
                                  * string option) (* function name        *)
                 | Predef_v   of string            (* predefined functions *)
                 | SpecForm_v of string            (* special form         *)
                 | Thunk_v    of exp * env         (* for lazy evaluation  *)
                 | Dyn_v      of exp               (* delayed dynamic scope*)

  and env       = Env of (string * value * typ) list

  val top_level = Env([

    (* Allows for the implementation of statement lists. *)
    ("stmtlst",   Predef_v "stmtlst",      Fn_t(Undef_t, Undef_t)),

    (* Compare the implementation of ncat and ncat2.     *)
    ("ncat",      Predef_v   "ncat",       Fn_t(Undef_t, String_t)),
    ("ncat2",     SpecForm_v "ncat2",      Fn_t(Undef_t, String_t)),

    ("lazylet",   SpecForm_v "lazylet",    Fn_t(Tuple_t([String_t,
                                                       Undef_t,
                                                       Undef_t]),
                                                Undef_t)),
    ("lazylet2",  SpecForm_v "lazylet2",   Fn_t(Tuple_t([Undef_t,
                                                       Undef_t,
                                                       Undef_t]),
                                                Undef_t)),
    ("lazylet3",  SpecForm_v "lazylet3",   Fn_t(Tuple_t([String_t,
                                                       Undef_t,
                                                       Undef_t]),
                                                Undef_t)),
    (* Note how one can allow for synonims.              *)
    ("lazysubst", SpecForm_v "lazylet",    Fn_t(Tuple_t([String_t,
                                                       Undef_t,
                                                       Undef_t]),
                                                Undef_t)),
    ("if3",       SpecForm_v  "if3",       Fn_t(Tuple_t([Bool_t,
                                                       Undef_t,
                                                       Undef_t]),
                                                Undef_t)),
    ("print",     Predef_v    "print",     Fn_t(String_t, Tuple_t([]))),
    ("fail",      Predef_v    "fail",      Fn_t(String_t, Tuple_t([]))),
    ("hd",        Predef_v    "hd",        Fn_t(List_t(Undef_t), Undef_t)),
    ("tl",        Predef_v    "tl",        Fn_t(List_t(Undef_t),
                                                List_t(Undef_t))),
    ("null",      Predef_v    "null",      Fn_t(List_t(Undef_t),Bool_t)),
    ("explode",   Predef_v    "explode",   Fn_t(String_t,List_t Char_t)),
    ("implode",   Predef_v    "implode",   Fn_t(List_t Char_t, String_t))
  ])


  fun lookupBinding (id: string, Env(en)): (value * typ) option =
    case List.find (fn ((s, _, _)) => id = s) en of
      NONE            => NONE
    | SOME((_, v, t)) => SOME((v, t))

  fun insertBinding (id: string, (v, t): value * typ, Env(en)): env =
    Env((id, v, t) :: en)

end
