(*
  The types used to build Mini-SML abstract syntax trees.
  Also, utility functions.
*)
structure AbstractSyntax = struct

  type id = string

  (* Mini-SML types - can't use type because it is a reserved word. *)
  datatype typ = Int_t
               | Real_t
               | Bool_t
               | Char_t
               | String_t
               | Tuple_t   of typ list
               | List_t    of typ
               | Fn_t      of (typ * typ)
               | Ref_t     of typ
               | Undef_t   (* undefined type for [] *)

  (*
    Compare two types. Note that because of Undef_t's, types might
    not be fully defined. The function below returns the most general
    unified type that corresponds to the two argument types, if such
    a unified type exists.
  *)
  exception TypeUnification

  fun unifyTypes (t: typ, t': typ): typ =
    case (t, t') of
      (Undef_t,      _       )         => t'
    | (_,            Undef_t )         => t
    | (Int_t,        Int_t   )         => Int_t
    | (Real_t,       Real_t  )         => Real_t
    | (Bool_t,       Bool_t  )         => Bool_t
    | (Char_t,       Char_t  )         => Char_t
    | (String_t,     String_t)         => String_t
    | (Tuple_t tl,   Tuple_t tl')      =>
       if List.length tl <> List.length tl' then raise TypeUnification
        else Tuple_t(ListPair.map (fn (ta,tb) => unifyTypes(ta,tb)) (tl, tl'))
    | (List_t tl,    List_t tl')       =>  List_t(unifyTypes(tl, tl'))
    | (Fn_t(fa, rt), Fn_t(fa', rt'))   =>
        Fn_t(unifyTypes(fa, fa'), unifyTypes(rt, rt'))
    | (Ref_t rt, Ref_t rt')            => Ref_t(unifyTypes(rt, rt'))
    | _                                => raise TypeUnification


  type funrec = {name:id, args: (id * typ) list, ret_typ: typ}

  datatype binop = Plus
               | Times
               | Minus
               | Mod
               | Div
               | Slash

               | Equal
               | Greater
               | Less
               | GreaterEq
               | LessEq

               | Cons
               | Append

               | AndAlso
               | OrElse

               | Concat

               | Assign

  datatype unop = Neg                            (* ~        *)
               |  Not                            (* not      *)
               |  Ref                            (* ref      *)
               |  Deref                          (* !        *)

  datatype exp = Int_c     of int                 (* 17      *)
               | Real_c    of real                (* 12.73   *)
               | Bool_c    of bool                (* true    *)
               | Char_c    of char                (* #"a"    *)
               | String_c  of string              (* "alpha" *)

               | Id_e      of id                  (* any variable identifier  *)

               | If_e      of (exp * exp * exp)   (* if b then a else b       *)
               | Let_e     of (decl list * exp)   (* let val x = 4 in x+x end *)

               | Fn_e      of ((id*typ)list * typ * exp)(* fn (s:int):int=>6 *)
               | Apply_e   of (exp * exp)         (* increment(6)       *)

               | Unop_e    of (unop * exp)        (* not true *)
               | Binop_e   of (exp * binop * exp) (* 5 + 5    *)

               | Tuple_e   of (exp list)          (* (5,4,4,3)  *)
               | Ith_e     of (int * exp)         (* #1 (3,4,5) *)

               | List_e    of (exp list)          (* [1, 2] *)



  and decl =     Val_d     of (id * typ * exp)    (* val x = (5,2)            *)
               | Fun_d     of (funrec * exp)      (* fun inc(i:int):int = i+1 *)


  datatype top_level =
                 Exp_t    of exp
               | Decl_t   of decl list

end
