open Naturals
open Big_int

module type FIELD = sig
  (** [t] is the type of the element in the field. *)
  type t

  (** [zero] is the additive identity *)
  val zero : t

  (** [one] is the multiplicative identity *)
  val one : t

  (** [add a b] returns [a + b] *)
  val add : t -> t -> t

  (** [mul a b] returns [a * b] *)
  val mul : t -> t -> t

  (** [neg x] returns [-x], the additive inverse of x *)
  val neg : t -> t

  (** [inv x] returns [1/x], the multiplicative inverse of x
   *  precondition: x is not equal to 0 *)
  val inv : t -> t

  (** [num_of_big_int b] returns the field representation of Big Int [b] *)
  val num_of_big_int : big_int -> t

  (** [num_of_int i] returns the field representation of int [i] *)
  val num_of_int : int -> t
end

module type OrderedField = sig
  include FIELD

  (** [cmp a b] returns an integer i, such that:
   *  if a is greater than b, then i>0
   *  if a is equal to b,     then i=0
   *  if a is less than b,    then i < 0
   *)
  val cmp : t -> t -> int
end

module Floats : (OrderedField with type t = float)

module type Q = sig
  include OrderedField

  (** [make p q] returns [Some x] where [x] represents the
      rational [p/q] if [q <> 0] and [None] otherwise. *)
  val make : big_int -> big_int -> t option

  (** [eq a b] returns [a = b]. *)
  val eq : t -> t -> bool

  (** [sub a b] returns [a - b]. *)
  val sub : t -> t -> t

  (** [div a b] returns [a / b].
   *  precondition: b is not zero *)
  val div : t -> t -> t

  (** [abs x] returns x if x is non-negetive and [neg x] otherwise. *)
  val abs : t -> t

  (** [ceil x] returns the smallest integer that is strictly greater than x *)
  val ceil : t -> t

  (** [max a b] returns a if a>=b, b otherwise. *)
  val max : t -> t -> t

  (** [gcd a b] returns the greatest common denominator of [a] and [b] *)
  val gcd : t -> t -> t

  (** [to_int x] returns (p,q) where x is a rational representing p/q
   *  it is undefined if p or q cannot be represented as an integers *)
  val to_int : t -> int * int

  (** [to_string x] returns a string version of your rational
    * For testing: implementation is up to you *)
  val to_string : t -> string
end

module Rationals : Q


module MakeReal (Nat:N) (Rat:Q) : sig
  include OrderedField

  (** [make_real_from_rational x] returns the
        representation of x as a real number *)
  val make_real_from_rational : Rat.t -> t

  (** [approx x n] returns a rational q such that |x-q|< 1/n *)
  val approx : t -> Nat.t -> Rat.t

(** We define reals as a regular sequence of rationals *)
end with type t = Nat.t -> Rat.t

module Reals : sig
  include OrderedField

(** [make_real_from_rational x] returns the
        representation of x as a real number *)
  val make_real_from_rational : Rationals.t -> t

  (** [approx x n] returns a rational q such that |x-q|< 1/n *)
  val approx : t -> BigIntNat.t -> Rationals.t
end with type t = BigIntNat.t -> Rationals.t


(** [sqrt x] returns the square root of x
    [precondition]: x >= 0 *)
val sqrt : Reals.t -> Reals.t


module FieldFunctions (F:OrderedField) : sig
  val sort_list : F.t list -> F.t list
end

module FloatFunctions : sig val sort_list : Floats.t list -> Floats.t list end
module RatFunctions : sig val sort_list : Rationals.t list -> Rationals.t list end
module RealFunctions : sig val sort_list : Reals.t list -> Reals.t list end
