open Big_int

type cmp =
  | Less
  | Greater
  | Equal

module type Q = sig
  type t
  val zero : t (** the rational number 0 *)
  val one : t  (** the rational number 1 *)

  (** [make p q] returns [Some x] where [x] represents the
      rational [p/q] if [q <> 0] and [None] otherwise. *)
  val make : int -> int -> t option

  (** make_big behaves exactly as make, except it takes arbitrary
      big integer as inputs. *)
  val make_big: Big_int.big_int -> Big_int.big_int -> t option

  (** [add a b] returns [a + b]. *)
  val add : t -> t -> t

  (** [inv x] returns [1/x]. *)
  val inv : t -> t

  (** [neg x] returns [-x]. *)
  val neg : t -> t

  (** [mul a b] returns [a * b]. *)
  val mul : t -> t -> t

  (** [eq a b] returns [a = b]. *)
  val eq : t -> t -> bool

  (** [cmp a b] returns [Less] if [a < b], [Greater] if
      [a > b], and [Equal] if [a = b]. *)
  val cmp : t -> t -> cmp

  (** [div a b] returns [a / b]. *)
  val div : t -> t -> t

  (** [gcd a b] returns the gcd of [a] and [b]. *)
  val gcd : t -> t -> t

  (** [abs x] returns x if x is non-negetive and [neg x] otherwise. *)
  val abs : t -> t

  (** [ceil x] returns the smallest integer that is strictly greater than x *)
  val ceil : t -> t

  (** [max x y] returns x if x>=y , y otherwise. *)
  val max : t -> t -> t

  (** [to_int x] returns [p] and [q] such that [p/q = x]. *)
  val to_int : t -> Big_int.big_int * Big_int.big_int

  (** [to_string x] returns the string representation of x as [num/denom]*)
  val to_string: t-> string
end


module Rational : Q = struct
  type t = (Big_int.big_int * Big_int.big_int)
  let ( * ) = Big_int.mult_big_int
  let (+) = Big_int.add_big_int
  let (-) = Big_int.sub_big_int
  let (/) = Big_int.div_big_int
  let big_int_of_int = Big_int.big_int_of_int

  let zero = (Big_int.zero_big_int, Big_int.unit_big_int)
  let one = (Big_int.unit_big_int,Big_int.unit_big_int)

  let make_big p q =
    if Big_int.eq_big_int q Big_int.zero_big_int then
      None
    else
      Some (p,q)

  let make p q =
    if q = 0 then
      None
    else
      Some (big_int_of_int p, big_int_of_int q)

  let add a b =
    let (an, ad) = a in
    let (bn, bd) = b in
    if (Big_int.compare_big_int ad bd) = 0 then
      (an + bn, ad)
    else
      (an * bd + ad * bn, ad * bd)

  let inv x =
    let (xn, xd) = x in
    (xd, xn)

  let neg x =
    let (xn, xd) = x in
    if Big_int.le_big_int xd Big_int.zero_big_int then (xn, (big_int_of_int (-1)) * xd)
    else ((big_int_of_int (-1)) * xn, xd)

  let mul a b =
    let (an, ad) = a in
    let (bn, bd) = b in
    (an * bn, ad * bd)

  let eq a b =
    let (an, ad) = a in
    let (bn, bd) = b in
    Big_int.compare_big_int (an * bd) (ad * bn) = 0

  let cmp a b =
    let (an, ad) = a in
    let (bn, bd) = b in
    let diff = Big_int.compare_big_int (an*bd) (ad*bn) in
    if diff = 0 then
      Equal
    else if diff > 0 then
      Greater
    else
      Less

  let div a b =
    mul a (inv b)

  let sub a b =
    add a (neg b)

  let abs (x:t) =
    match cmp x zero with
    | Less -> neg x
    | _ -> x

  let rec gcd_help a b =
    if Big_int.eq_big_int b Big_int.zero_big_int then a
    else gcd_help b (Big_int.mod_big_int a b)

  let reduce (p,q) =
    let gcd = gcd_help p q in
    (Big_int.div_big_int p gcd, Big_int.div_big_int q gcd)

  let lcm m n =
    if Big_int.eq_big_int m Big_int.zero_big_int then Big_int.zero_big_int
    else Big_int.div_big_int
      (Big_int.abs_big_int (Big_int.mult_big_int m n)) (gcd_help m n)

  let rec gcd (p1,q1) (p2,q2) =
    let (p1',q1') = reduce (p1,q1) in
    let (p2',q2') = reduce (p2,q2) in
    (gcd_help p1' p2', lcm q1' q2')

  let to_int x =
    let (xn, xd) = x in
    let big_one = big_int_of_int 1 in
    let (d, _) = gcd (xn, big_one) (xd, big_one) in
    (xn / d, xd / d)

(** Exercise 1. *)
(************** TODO ****************)
  let ceil x =
    failwith "unimplemented"

  let max x y =
    failwith "unimplemented"
(************** End TODO ****************)

  let to_string (p,q) =
    let p,q = to_int (p,q) in
    let p,q = (Big_int.string_of_big_int p, Big_int.string_of_big_int q) in
    Printf.sprintf "%s/%s" p q
end
