(* OBJECT ENCODING
 * CS 3110 Spring 2015
 * Author: Michael Clarkson
 * Based on:  Types and Programming Languages, Chapter 18,
 *   by Benjamin C. Pierce. *)

(***************************************************
 * REPRESENTATION 
 ***************************************************)

(* Objects can be represented as a data structure 
 * that encapsulates some internal state and offers
 * access to that state through methods. The internal
 * state naturally is not exposed in the type. Rather,
 * the type of the object gives the publicly accessible
 * interface. *)

type counter = {
  get : unit -> int;
  inc : unit -> unit;
}

(***************************************************
 * ENCAPSULATION
 ***************************************************)

(* [c] is an object with internal state [x],
 * which cannot be accessed (or even named)
 * from outside. *)
let c : counter =
  let x = ref 0 in {
    get = (fun () -> !x);
    inc = (fun () -> x := !x+1);
  }

let _ = assert (c.inc(); c.inc(); c.get() = 2)

(* We can write functions that manipulate objects. *)
    
let inc3 (c:counter) = c.inc(); c.inc(); c.inc()

let _ = assert (inc3 c; c.get() = 5)

(* An object "constructor" *)

let new_counter = fun () ->
  let x = ref 0 in {
    get = (fun () -> !x);
    inc = (fun () -> x := !x+1);
  }

let d = new_counter()
let e = new_counter()
let _ = assert(d.inc(); d.get() = 1)
let _ = assert(e.get() = 0)

(***************************************************
 * SUBTYPING
 ***************************************************)

(* Suppose we create another kind of counter that
 * has an additional method [reset]. *)

type reset_counter = {
  get : unit -> int;
  inc : unit -> unit;
  reset : unit -> unit;
}

let new_reset_counter () =
  let x = ref 0 in {
    get = (fun () -> !x);
    inc = (fun () -> x:=!x+1);
    reset = (fun () -> x:=0);
  }

let rc = new_reset_counter()
let _ = assert(rc.inc(); rc.reset(); rc.inc(); rc.get() = 1)

(* A [reset_counter] can be treated as a [counter] by
 * repackaging it into a smaller record.  The following
 * "coercion function" can be thought of as an upcast from
 * a subtype to a supertype. *)

let counter__of__reset_counter (rc : reset_counter) = {
  get = rc.get;
  inc = rc.inc;
}

(* Thinking of [counter] as the supertype and [reset_counter]
 * as the subtype, upcasting lets us apply a function defined
 * on the supertype to an object of a subtype. *)

let _ = assert (inc3 (counter__of__reset_counter rc); rc.get() = 4)

(***************************************************
 * INHERITANCE
 ***************************************************)

(* Let's think of a "class" as a way of generating related
 * objects.  Specifically, a class is a function from the
 * internal representation of an object's state to the object. 
 * Thus, a class produces a collection of methods, all of 
 * which use the same state. *)

(* We'll make the representation a record, so that it's
 * easy to represent objects with more than one piece of 
 * internal state. *)
type counter_rep = {
  x : int ref;
}

let counter_class = fun (r:counter_rep) -> {
  get = (fun () -> !(r.x));
  inc = (fun () -> (r.x := !(r.x) + 1));
}

(* A "constructor" is a function that uses the class
 * to produce a new object. *)

let new_counter () =
  let r = {x = ref 0} in
  counter_class r

(* A class can "inherit" from another class by reusing the
 * methods from the "superclass" and adding additional methods. *)

let reset_counter_class = fun (r:counter_rep) -> 
  let super = counter_class r in {
    get = super.get;
    inc = super.inc;
    reset = (fun () -> r.x := 0)
  }

let new_reset_counter () =
  let r = {x=ref 0} in
  reset_counter_class r

let rc = new_reset_counter()
let _ = assert(rc.inc(); rc.reset(); rc.get() = 0)

(* A class can also add additional "instance variables" by 
 * defining a larger rep. *)

type backup_counter_rep = {
  x : int ref;
  b : int ref;
}

(* But we also need a way to "upcast" a rep for the subclass
 * into a rep for the superclass. *)

let counter_rep__of__backup_counter_rep (r : backup_counter_rep) = {
  x = r.x;
}

(* By the way, the above upcast is where we need the 
 * representations to be records whose fields are refs, 
 * rather than records of mutable fields. The latter
 * wouldn't allow us to make the two reps refer to the
 * same cell. *)

(* [backup_counter] adds a [backup] method that saves the current
 * value of the counter.  [reset] will restore that value.  
 * [backup_counter] will be a subclass of [reset_counter]. *)

type backup_counter = {
  get : unit -> int;
  inc : unit -> unit;
  reset : unit -> unit;
  backup : unit -> unit
}

(* [backup_counter_class] extends [reset_counter_class] with a new
 * method [backup] and also overrides the existing method [reset]. *)

let backup_counter_class (r : backup_counter_rep) =
  let super = reset_counter_class (counter_rep__of__backup_counter_rep r) in {
    get = super.get;
    inc = super.inc;
    reset = (fun () -> r.x := !(r.b));
    backup = (fun () -> r.b := !(r.x));
  }

let new_backup_counter () =
  let r = {x = ref 0; b = ref 0} in
  backup_counter_class r

let bc = new_backup_counter()
let _ = assert (bc.inc(); bc.backup(); bc.inc(); bc.reset(); bc.get() = 1)

