(* Univariate polynomials with integer coefficients. *)

module type POLYNOMIAL = sig
  type poly
  val zero : poly
  (* monomial (c, d) is the polynomial cx^d. Requires d >= 0. *)
  val monomial : int * int -> poly
  (* degree is the largest exponent with a nonzero coefficient *)
  val degree : poly -> int
  (* evaluate p at integer value x *)
  val evaluate : poly * int -> int
  (* coeff (p, d) is the coefficient of the of the degree-d term,
   * or zero if there is no such term. Requires: d >= 0. *)
  val coeff : poly * int -> int
  (* ring operations *)
  val plus : poly * poly -> poly
  val minus : poly * poly -> poly
  val times : poly * poly -> poly
  (* conversion to printable form *)
  val to_string: poly -> string
end

module Polynomial : POLYNOMIAL = struct
  (* Univariate polynomials represented as a list of coefficients.
   * Degree is based on position in the list, low degrees first. *)
  type poly = int list
  let zero = []

  (* A monomial cx^d is a list of length d with c last and all
   * other elements 0 *)
  let rec monomial (coeff, degree) =
    if degree < 0 then failwith "negative degree" else
    match (coeff, degree) with
      | (0, _) -> zero
      | (c, 0) -> [c]
      | (c, d) -> 0 :: monomial (c, d - 1)

  let degree p =       
    match p with
      | [] -> 0
      | _ -> List.length p - 1

  let rec coeff (p, n) =
    match p with
      | [] -> 0
      | h :: t -> if n = 0 then h else coeff (t, n - 1)

  (* plus and minus both operate on two polynomials term by term,
   * so this function abstracts out the common pattern *)
  let rec termapply (f, p, q) =
    match (p, q) with
      | ([], []) -> []
      | ([], b :: q2) -> f (0, b) :: termapply (f, [], q2)
      | (a :: p2, []) -> f(a, 0) :: termapply (f, p2, [])
      | (a :: p2, b :: q2) -> f (a, b) :: termapply (f, p2, q2)

  let plus (p, q) = termapply ((fun (a, b) -> a + b), p, q)
  let minus (p, q) = termapply ((fun (a, b) -> a - b), p, q)
  let times (p, q) = failwith "not implemented"

  let evaluate (p, x) =
    List.fold_right (fun h a -> h + x * a) p 0

  let to_string p =
    let term (d, s) c =
      if c = 0 then (d + 1, s) else 
      (d + 1, (if s = "" then "" else s ^ " + ") ^
      (if c = 1 then "" else (if c = (-1) then "-" else string_of_int c)) ^
      (if d > 0 then "x" ^ (if d > 1 then "^" ^ string_of_int d else "")
      else "")) in
    match List.fold_left term (0, "") p with (d, s) -> s

end

(* some test cases *)

let px2 = Polynomial.monomial (1, 2);;
let p3x2 = Polynomial.monomial (3, 2);;
let p2x3 = Polynomial.monomial (2, 3);;
let p3x4 = Polynomial.monomial (3, 4);;
let px5 = Polynomial.monomial (1, 5);;
let tp1 = Polynomial.plus (px2, px2);;
let tp2 = Polynomial.plus (tp1, Polynomial.plus (p2x3, px5));;
Polynomial.evaluate (tp2, 2);;
Polynomial.to_string tp2;;
