#use "regexp_util.ml";;

module Util = struct
  open List

  (* List of contiguous integers: 1 <|> 5 = [1; 2; 3; 4; 5] *)
  let (<|>) x y =
    let rec loop i acc = if i > y then rev acc else loop (i+1) (i::acc) in
    loop x []

  let range n = 0 <|> n-1

  let (^^) f = fun l -> concat (map f l)

  (* Takes a string and returns a list of characters *)
  let explode str =
    let rec expl i l = if i < 0 then l else expl (i-1) (str.[i] :: l) in
    expl (String.length str - 1) []
  let implode s = String.concat "" (List.map (fun c -> String.make 1 c) s)

  let alphabet = explode "abcdefghijklmnopqrstuvwxyz"

  (* Removes duplicates from a list (useful for epsilon_closure) *)
  let unique l =
    let uniq e = function
      | [] -> [e]
      | h::_ as l -> if e = h then l else e :: l in
    fold_right uniq (sort compare l) []
end

(* Immutable 2D tables *)
module Table
  : sig
    type 'a t
    (* Vertical dimension, horizontal dimension, default value *)
    val create : int -> int -> 'a -> 'a t

    (* Table, row index, column index, new value *)
    val add : 'a t -> int -> int -> 'a -> 'a t
    val lookup : 'a t -> int -> int -> 'a
    val map : ('a -> 'b) -> 'a t -> 'b t

    (* Concatenates the rows of two tables *)
    val concat : 'a t -> 'a t -> 'a t

    (* Adds an element to a table of lists *)
    val append : 'a list t -> int -> int -> 'a -> 'a list t

    val height : 'a t -> int
    val width : 'a t -> int
  end

  = struct
    type 'a t = 'a array array

    let create m n v =
      let rows = List.map (fun _ -> Array.make n v) (Util.range m) in
      Array.of_list rows

    let add t m n v =
      let new_row = Array.copy t.(m) in
      let new_t   = Array.copy t in
      new_row.(n) <- v;
      new_t.(m) <- new_row;
      new_t

    let lookup t m n = try t.(m).(n) with _ -> Printf.printf "%d %d\n" m n; t.(m).(n)
    let map f t = Array.map (Array.map f) t
    let concat = Array.append
    let append t m n v =
      let old = lookup t m n in
      add t m n (v :: old)

    let height t = Array.length t
    let width t = Array.length t.(0)
  end

module NFA
  : sig
    type state = int
    type nfa
    type transition = state * char option * state

    (* Assumes state list is [0..n_states-1] *)
    val make :
      start:state list ->
      accept:state list ->
      n_states:int ->
      alphabet:char list ->
      transitions:transition list -> nfa
    val count_states : nfa -> int
    val lookup : nfa -> state -> char option -> state list
    val accept : nfa -> string -> bool

    (* All operations are done immutably *)
    val union : nfa -> nfa -> nfa
    val concat : nfa -> nfa -> nfa
    val star : nfa -> nfa
  end

  = struct
    open Util
    type state = int
    type transition = state * char option * state

    module StateSet = Set.Make
      (struct
        type t = state
        let compare = compare
       end)

    module Dict = Map.Make
      (struct
        type t = char
        let compare = compare
       end)

    type nfa = {table : state list Table.t;
                start : state list;
                final : StateSet.t;
                alphabet : int Dict.t}

    (* Epsilon (as None) always maps to zero *)
    let char_to_int alphabet = function
      | None -> 0
      | Some c -> Dict.find c alphabet

    let add_trans alphabet table (s0, c, s1) =
      let n = char_to_int alphabet c in
      Table.append table s0 n s1
    let lookup nfa state c =
      Table.lookup (nfa.table) state (char_to_int nfa.alphabet c)

    (* Assumes states are integers in [0..n_states-1] *)
    let make ~start ~accept ~n_states ~alphabet ~transitions =
      failwith "NFA.make: Not implemented."

    let count_states nfa =
      failwith "NFA.count_states: Not implemented."

    let epsilon_closure nfa states =
      failwith "NFA.epsilon_closure: Not implemented."

    (* Precondition: current_states is epsilon_closed *)
    let step nfa current_states char =
      failwith "NFA.step: Not implemented."

    let accept nfa string =
      failwith "NFA.accept: Not implemented."

    (* Assumes alphabets are the same *)
    let union nfa1 nfa2 =
      failwith "NFA.union: Not implemented."

    (* Adds epsilon transitions from accepts of nfa1 to starts of nfa2 *)
    let concat nfa1 nfa2 =
      failwith "NFA.concat: Not implemented."

    (* Makes every start state a final state and adds epsilon transitions *)
    (* from every accept state to every start state *)
    let star nfa =
      failwith "NFA.star: Not implemented."
  end

module Levenshtein_Automata
  = struct
    open Util

    (* Accepts strings of edit distance strictly less than k_max *)
    let make string k_max =
      failwith "Levenshtein_Automata.make: Not implemented."
  end

module Regexp
  : sig
    type regexp
    val build : string -> regexp
    val accept : regexp -> string -> bool
  end
  = struct
    open Util
    open RegexpUtil
    type regexp = NFA.nfa

    let build str = failwith "Regexp.build: Not implemented."
    let accept _ = failwith "Regexp.accept: Not implemented."
  end
