(* Using higher-order functions to memoize computations. *)

(* Computes the least fixpoint of a given monotone functional *)
let fix (t : (('a -> 'b) -> 'a -> 'b)) : 'a -> 'b =
  let rec g x = t g x in g

(* The same, but with memoization by a hashtable. *)
let fix_memo (t : ('a -> 'b) -> 'a -> 'b)) : 'a -> 'b =
  let memo = Hashtbl.create 11 in
  let rec g x =
    try Hashtbl.find memo x
    with Not_found ->
      let y = t g x in
      Hashtbl.add memo x y; y
  in g
  
(* a monotone functional that "unwinds" the Fibonacci definition *)
(* the actual fib function is the least fixpoint of t_fib *)
let t_fib (g : int -> int) (n : int) : int =
  if n < 2 then 1 else g (n-1) + g (n-2)

(* the fibonacci function without memoization *)
let fib : int -> int = fix t_fib

(* the fibonacci function with memoization *)
let fib_memo : int -> int = fix_memo t_fib
