(*  Determine the best way to break a list of words with lengths in the
 *  list "lengths" into a series of lines with maximum length "target".
 *  Different possible ways of doing this are compared based on the sum
 *  of the cubes of the difference between the target width and the actual
 *  width; this has the effect of tending to equalize line lengths.
 *)
let cube (x : int) = x * x * x
let big = 10000

(* Result of formatting a string. A result (lst, n)
   means a string was formatted into the lines in lst,
   with a total sum-of-cubes cost of n. *)
type breakResult = string list * int

(* Result: format the words in "words" into a list of lines optimally,
 * minimizing the sum of the cubes of differences between the line lengths
 * and "target".
 * Performance: worst-case time is exponential in the number of words.
 *)
let linebreak1 (words : string list) (target : int) : string list =
  let rec lb (clen : int) (words : string list) : breakResult =
    match words with
      [] -> ([""], 0) (* no charge for last line *)
    | word :: rest ->
      (* Try two ways of doing it: (1) insert a linebreak right after
       * current word, or (2) continue the current line.  Pick the
       * better one. *)
      let wlen = String.length word in
      let contlen = if clen = 0 then wlen else clen + 1 + wlen in
      let (l1, c1') = lb 0 rest in
      let c1 = c1' + cube (target - contlen) in
      if contlen <= target
      then
        let (h2 :: t2, c2) = lb contlen rest in
        if c1 < c2 then (word :: l1, c1)
                   else ((if h2 = ""
                          then word
                          else word ^ " " ^ h2) :: t2, c2)
      else (word :: l1, big) in
  let (result, cost) = lb 0 words in
  result
