
open Util

(* character encoding *)
let utf8 = ref false

type id = string

type term = Term of id * term list

type formula =
  | Rel of id * term list
  | False
  | True
  | And of formula * formula
  | Or of formula * formula
  | Not of formula
  | Imp of formula * formula
  | Iff of formula * formula
  | Exists of id * formula
  | Forall of id * formula

let rec term_to_string (Term (f, t) : term) : string = f ^ terms_to_string t

and terms_to_string (t : term list) : string =
  match t with
    | [] -> ""
    | _ -> Printf.sprintf "(%s)" (String.concat "," (List.map term_to_string t))

let op_to_string = function
  | Rel (s, _) -> s
  | False -> if !utf8 then "⊥" else "0"
  | True -> if !utf8 then "T" else "1"
  | And _ -> if !utf8 then "∧" else "&"
  | Or _ -> if !utf8 then "∨" else "|"
  | Not _ -> if !utf8 then "¬" else "~"
  | Imp _ -> if !utf8 then "⇒" else "->"
  | Iff _ -> if !utf8 then "⇔" else "<->"
  | Exists _ -> if !utf8 then "∃" else "E"
  | Forall _ -> if !utf8 then "∀" else "A"

let op_to_latex = function
  | Rel (s, _) -> s
  | False -> "\\bot"
  | True -> "\\top"
  | And _ -> "\\wedge"
  | Or _ -> "\\vee"
  | Not _ -> "\\neg "
  | Imp _ -> "\\Rightarrow"
  | Iff _ -> "\\Leftrightarrow"
  | Exists _ -> "\\exists "
  | Forall _ -> "\\forall "

let op_precedence = function
  | Rel _ -> 6
  | True -> 6
  | False -> 6
  | Not _ -> 5
  | Exists _ -> 5
  | Forall _ -> 5
  | And _ -> 4
  | Or _ -> 3
  | Imp _ -> 2
  | Iff _ -> 1

let rec to_ (op_to_ : formula -> string) (e : formula) : string =
  let to_ = to_ op_to_ in
  let s0 = op_to_ e in
  let p0 = op_precedence e in
  let paren s = Printf.sprintf "(%s)" s in
  let infix s1 op s2 = Printf.sprintf "%s %s %s" s1 op s2 in
  match e with
    | Rel (p, t) -> s0 ^ terms_to_string t
    | (False | True) -> s0
    | (And (e1, e2) | Or (e1, e2) | Iff (e1, e2)) ->
        (* left-associative operators *)
        let p1 = op_precedence e1 in
        let p2 = op_precedence e2 in
        let s1 = if p1 < p0 then paren (to_ e1) else to_ e1 in
        let s2 = if p2 <= p0 then paren (to_ e2) else to_ e2 in
        infix s1 s0 s2
    | Imp (e1, e2) ->
        (* Imp is right associative *)
        let p1 = op_precedence e1 in
        let p2 = op_precedence e2 in
        let s1 = if p1 <= p0 then paren (to_ e1) else to_ e1 in
        let s2 = if p2 < p0 then paren (to_ e2) else to_ e2 in
        infix s1 s0 s2
    | Not e1 ->
        let p1 = op_precedence e1 in
        let s1 = if p1 < p0 then paren (to_ e1) else to_ e1 in
        s0 ^ s1
    | (Exists (x, e1) | Forall (x, e1)) ->
        let p1 = op_precedence e1 in
        let s1 = if p1 < p0 then paren (to_ e1) else to_ e1 in
        s0 ^ x ^ "." ^ s1
        
let to_string : formula -> string = to_ op_to_string
let to_latex : formula -> string = to_ op_to_latex

(* check arities of symbols *)
let typecheck (e : formula) : unit =
  let h = Hashtbl.create 11 in
  let check_one x n =
    try if Hashtbl.find h x <> n then failwith ("type error: " ^ x)
    with Not_found -> Hashtbl.add h x n in
  let rec check_arity id subterms =
    check_one id (List.length subterms);
    List.iter (fun (Term (x, t)) -> check_arity x t) subterms in
  let rec check (e : formula) : unit =
    match e with
    | Rel (id, t) -> check_arity id t
    | (False | True) -> ()
    | (And (e1, e2) | Or (e1, e2) | Iff (e1, e2) | Imp (e1, e2)) -> check e1; check e2
    | Not e1 -> check e1
    | (Exists (x, e1) | Forall (x, e1)) -> check_one x 0; check e1
  in check e

(* get free variables of a list of formulas *)
let fv (s : formula list) : id HashSet.t =
  let h = HashSet.make() in
  let rec fv_in_term (bv : id list) (Term (x, s) : term) : unit =
    if List.exists ((=) x) bv then ()
    else HashSet.add h x;
    List.iter (fv_in_term bv) s in
  let rec fv_in_formula (bv : id list) (e : formula) : unit =
    match e with
    | Rel (_, t) -> List.iter (fv_in_term bv) t
    | (False | True) -> ()
    | (And (e1, e2) | Or (e1, e2) | Iff (e1, e2) | Imp (e1, e2)) ->
        fv_in_formula bv e1; fv_in_formula bv e2
    | Not e1 -> fv_in_formula bv e1
    | (Exists (x, e1) | Forall (x, e1)) -> fv_in_formula (x :: bv) e1
  in List.iter (fv_in_formula []) s; h
  
(* get a fresh variable not occurring in a list of formulas *)
let fresh (s : formula list) : term = Term (Fresh.next (Fresh.make (fv s)), [])
