(* matching *)

open Ast

let no_match n = failwith ("Premise and conclusion do not match [" ^ string_of_int n ^ "]")

(* substitute term s for all occurrences of x in a term *)
let rec subst_in_term (s : term) (x : id) (Term (f, u) : term) : term =
  if x = f && u = [] then s
  else Term (f, List.map (subst_in_term s x) u)

(* substitute term s for all free occurrences of x in a formula e *)
let rec subst_in_formula (s : term) (x : id) (e : formula) : formula =
  let f = subst_in_formula s x in
  match e with
    | Rel (y, t) -> Rel (y, List.map (subst_in_term s x) t)
    | And (e1, e2) -> And (f e1, f e2)
    | Or (e1, e2) -> Or (f e1, f e2)
    | Iff (e1, e2) -> Iff (f e1, f e2)
    | Imp (e1, e2) -> Imp (f e1, f e2)
    | Not e1 -> Not (f e1)
    | Exists (y, e1) -> if x = y then e else Exists (y, f e1) 
    | Forall (y, e1) -> if x = y then e else Forall (y, f e1) 
    | (True | False) -> e

(* find a match for x in t that unifies s and t *)
let rec match_in_term (x : id) (s : term) (t : term) : term option =
  match (s, t) with
  | (Term (y, []), Term (z, [])) ->
      if x = y then Some t
      else if y = z then None
      else no_match 1
  | (Term (y, []), _) ->
      if x = y then Some t
      else no_match 2
  | (Term (f, s'), Term (g, t')) ->
      if f = g then match_in_terms x s' t'
      else no_match 3

(* find a match for x in some ti that unifies si and ti for all i *)
and match_in_terms (x : id) (ss : term list)  (tt : term list) : term option =
  match (ss, tt) with
  | ([], []) -> None
  | (s :: ss', t :: tt') ->
      let m = match_in_terms x ss' tt' in
      (match m with
        | None -> match_in_term x s t
        | Some u -> if subst_in_term u x s = t then m else no_match 4)
  | _ -> no_match 5

(* find a match for x in e that unifies d and e *)
let rec match_in_formula (x : id) (d : formula) (e : formula) : term option =
  match (d, e) with
    | (Rel (p, s), Rel (q, t)) ->
        if p = q then match_in_terms x s t
        else no_match 6
    | ((And (d1, d2), And (e1, e2)) | (Or (d1, d2), Or (e1, e2)) | (Iff (d1, d2), Iff (e1, e2)) | (Imp (d1, d2), Imp (e1, e2))) ->
        match_in_formulas x [d1; d2] [e1; e2]
    | (Not d1, Not e1) -> match_in_formula x d1 e1
    | ((Exists (y, d1), Exists (z, e1)) | (Forall (y, d1), Forall (z, e1))) ->
        if y <> z then no_match 7
        else if x <> y then match_in_formula x d1 e1
        else if d1 = e1 then None
        else no_match 8
    | ((True, True) | (False, False)) -> None
    | _ -> no_match 9

(* find a match for x in some ei that unifies di and ei for all i *)
and match_in_formulas (x : id) (ss : formula list) (tt : formula list) : term option =
  match (ss, tt) with
  | ([], []) -> None
  | (s :: ss', t :: tt') ->
      let m = match_in_formulas x ss' tt' in
      (match m with
        | None -> match_in_formula x s t
        | Some u -> if subst_in_formula u x s = t then m else no_match 10)
  | _ -> no_match 11

