
open Util

(* length of name files *)
let n = 1000
(* number of profiles to create *)
let m = 1000

(* read in and parse categories *)
let (category_names, categories) =
  try
    let c : string list = read_file "resources/categories.txt" in
    let f = Str.split (Str.regexp "[ ]*:[ ]*") in
    let c : string list list = transpose (List.map f c) in
    let (cat_names, c) =
      match c with
      | [c1; c2] -> (c1, c2)
      | _ -> failwith "1" in
    let f = Str.split (Str.regexp "[ ]*,[ ]*") in
    let c : string list list = List.map f c in
    let f = List.map (Str.split (Str.regexp "[ ]+")) in
    let c : string list list list = List.map f c in
    let f x =
      match x with
      | [attr; value] -> (attr, int_of_string value)
      | _ -> failwith "2" in
    let c : (string * int) list list = List.map (List.map f) c in
    let cat_values : (string * int) array list = List.map Array.of_list c in
    (cat_names, cat_values)
  with Failure s -> failwith (Printf.sprintf "malformed categories.txt (%s)" s)
  | exc -> failwith "malformed categories.txt (3)"

(* compute cumulative weights *)
let _ =
  let f (a : (string * int) array) =
    Array.iteri (fun r x -> if r = 0 then () else a.(r) <- (fst a.(r), snd a.(r) + snd a.(r-1))) a in
  List.iter f categories;
  List.iter (fun a -> if snd a.(Array.length a - 1) <> 100 then failwith "malformed categories.txt (4)") categories

(* read in name files *)
let lastnames = Array.of_list (read_file "resources/lastnames.txt")
let girlsfirstnames = Array.of_list (read_file "resources/girlsfirstnames.txt")
let boysfirstnames = Array.of_list (read_file "resources/boysfirstnames.txt")

(* create a random age and age range preference for partner *)
(* the age of the person is between 18 and 45, inclusive *)
(* the age range preference is a randomly chosen interval *)
(* between 3 and 10 years less than the person's age and between 3 and 10 *)
(* years more than the person's age, but not less than 18 nor more than 45 *)
let lo a = min a (max 18 (a - (Random.int 7 + 3)))
let hi a = max a (min 45 (a + (Random.int 7 + 3)))
let random_age () : string =
  let nb = Random.int 28 + 18 in
  Printf.sprintf "@%d@%d@%d" nb (lo nb) (hi nb)

(* construct a random non-sex-specific profile from the categories *)
let random_nss_profile () : string =
  let f (a : (string * int) array) =
    let r = Random.int 100 in
    let rec find i =
      if r < snd a.(i) then i else find (i+1) in
    fst a.(find 0) in
  let (rp : string list) = List.map f categories in
  let (rp : string list) = List.map (Printf.sprintf "@%s") rp in
  String.concat "" rp

(* construct a random name *)
let random_name (firstnames : string array) : string =
  Printf.sprintf "%s@%s" firstnames.(Random.int n) lastnames.(Random.int n)

(* create a random profile - boolean parameter determines sex *)
let random_profile (male : bool) : string =
  let sex = if male then "@M" else "@F" in
  let firstnames = if male then boysfirstnames else girlsfirstnames in
  let name = random_name firstnames in
  let age = random_age () in
  let profile = random_nss_profile () in
  Printf.sprintf "%s%s%s%s" name sex age profile

let _  =
  Random.self_init();
  assert (Array.length lastnames = n);
  assert (Array.length girlsfirstnames = n);
  assert (Array.length boysfirstnames = n);

  (* create profiles *)
  let rec all k sex profiles =
    if k = 0 then profiles else
    all (k - 1) sex (random_profile sex :: profiles) in
  let boys = all m true [] in
  let girls = all m false [] in
  write_file "out/males.txt" boys;
  write_file "out/females.txt" girls;
  write_file "out/profiles.txt" (boys @ girls)
