open Ast

(* Natural deduction proof *)

type proof =
  | Iff_intro of formula * proof * proof
  | Arrow_intro of formula * proof * proof
  | Arrow_elim of formula * proof * proof
  | And_intro of formula * proof * proof
  | And_elim_left of formula * proof
  | And_elim_right of formula * proof
  | Or_intro_left of formula * proof
  | Or_intro_right of formula * proof
  | Or_elim of formula * proof * proof * proof
  | Not_intro of formula * proof
  | Not_elim of formula * proof
  | Exists_intro of term * formula * proof
  | Exists_elim of term * formula * proof * proof
  | Forall_intro of term * formula * proof
  | Forall_elim of term * formula * proof
  | Efq of formula * proof
  | Magic of formula
  | Raa of formula * proof * proof
  | Unit
  | Assumption of int * formula
  | Task of int

(* substitute proof e for task t in proof p *)
let rec subst (e : proof) (t : int) (p : proof) : proof =
  let f = subst e t in
  match p with
  | Iff_intro (e, p1, p2) -> Iff_intro (e, f p1, f p2) 
  | Arrow_intro (e, p1, p2) -> Arrow_intro (e, f p1, f p2) 
  | Arrow_elim (e, p1, p2) -> Arrow_elim (e, f p1, f p2)
  | And_intro (e, p1, p2) -> And_intro (e, f p1, f p2)
  | And_elim_left (e, p1) -> And_elim_left (e, f p1)
  | And_elim_right (e, p1) -> And_elim_right (e, f p1)
  | Or_intro_left (e, p1) -> Or_intro_left (e, f p1)
  | Or_intro_right (e, p1) -> Or_intro_right (e, f p1)
  | Or_elim (e, p1, p2, p3) -> Or_elim (e, f p1, f p2, f p3)
  | Not_intro (e, p1) -> Not_intro (e, f p1)
  | Not_elim (e, p1) -> Not_elim (e, f p1)
  | Exists_intro (t, e, p1) -> Exists_intro (t, e, f p1)
  | Exists_elim (t, e, p1, p2) -> Exists_elim (t, e, f p1, f p2)
  | Forall_intro (t, e, p1) -> Forall_intro (t, e, f p1)
  | Forall_elim (t, e, p1) -> Forall_elim (t, e, f p1)
  | Efq (e, p1) -> Efq (e, f p1)
  | Magic e -> Magic e
  | Raa (e, p1, p2) -> Raa (e, f p1, f p2)
  | Unit -> Unit
  | Assumption (x, e) -> Assumption (x, e)
  | Task x -> if x = t then e else Task x

let rec to_latex (p : proof) : string =
  let var_to_latex x = Printf.sprintf "x_{%d}" x in
  let task_to_latex t = Printf.sprintf "t_{%d}" t in
  let premise0 (e : formula) (s : string) : string =
    Printf.sprintf "\\AxiomC{}\n\\RightLabel{\\tiny(%s)}\n\\UnaryInfC{$%s$}"
    s (Ast.to_latex e) in 
  let premise1 (e : formula) (p : proof) (s : string) : string =
    Printf.sprintf "%s\n\\RightLabel{\\tiny(%s)}\n\\UnaryInfC{$%s$}"
    (to_latex p) s (Ast.to_latex e) in 
  let premise2 (e : formula) (p1 : proof) (p2 : proof) (s : string) : string =
    Printf.sprintf "%s\n%s\n\\RightLabel{\\tiny(%s)}\n\\BinaryInfC{$%s$}"
    (to_latex p1) (to_latex p2) s (Ast.to_latex e) in 
  let premise3 (e : formula) (p1 : proof) (p2 : proof) (p3 : proof) (s : string) : string =
    Printf.sprintf "%s\n%s\n%s\n\\RightLabel{\\tiny(%s)}\n\\TrinaryInfC{$%s$}"
    (to_latex p1) (to_latex p2) (to_latex p3) s (Ast.to_latex e) in 
  match p with
  | Iff_intro (e, p1, p2) -> premise2 e p1 p2 "$\\Leftrightarrow$I"
  | Arrow_intro (e, Assumption (x, _), p1) -> premise1 e p1 ("$\\Rightarrow$I/$" ^ var_to_latex x ^ "$")
  | Arrow_elim (e, p1, p2) -> premise2 e p1 p2 "$\\Rightarrow$E"
  | And_intro (e, p1, p2) -> premise2 e p1 p2 "$\\wedge$I"
  | And_elim_left (e, p1) -> premise1 e p1 "$\\wedge$EL"
  | And_elim_right (e, p1) -> premise1 e p1 "$\\wedge$ER"
  | Or_intro_left (e, p1) -> premise1 e p1 "$\\vee$IL"
  | Or_intro_right (e, p1) -> premise1 e p1 "$\\vee$IR"
  | Or_elim (e, p1, p2, p3) -> premise3 e p1 p2 p3 "$\\vee$E"
  | Not_intro (e, p1) -> premise1 e p1 "$\\neg$I"
  | Not_elim (e, p1) -> premise1 e p1 "$\\neg$E"
  | Exists_intro (t, e, p1) -> premise1 e p1 "$\\exists$I"
  | Exists_elim (t, e, p1, p2) -> premise2 e p1 p2 "$\\exists$E"
  | Forall_intro (t, e, p1) -> premise1 e p1 "$\\forall$I"
  | Forall_elim (t, e, p1) -> premise1 e p1 "$\\forall$E"
  | Efq (e, p1) -> premise1 e p1 "EFQ"
  | Magic e -> premise0 e "MAGIC"
  | Raa (e, Assumption (x, _), p1) -> premise1 e p1 ("RAA/$" ^ var_to_latex x ^ "$")
  | Unit -> premise0 True "UNIT" 
  | Assumption (x, e) ->
      Printf.sprintf "\\AxiomC{$%s\\mathrel{:}%s$}" (var_to_latex x) (Ast.to_latex e)
  | Task t -> Printf.sprintf "\\AxiomC{$%s$}" (task_to_latex t)
  | _ -> failwith "Bad proof"
